/**
 *******************************************************************************
 * @file    dpm32m0xx_cct.h
 *
 * @brief   Header file for CCT firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_CCT_H__
#define __DPM32M0XX_CCT_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"

/**
 ******************************************************************************
 * @brief CCT input capture analog compare channel enumeration.
 *****************************************************************************/
typedef enum
{
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036) || defined (DPM32M030)
  CCT_IC_ACMP0           = 0x00UL,   /*!< Analog compare channel 0. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  CCT_IC_ACMP1           = 0x01UL,   /*!< Analog compare channel 1. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  CCT_IC_ACMP2           = 0x02UL,   /*!< Analog compare channel 2. */
  CCT_IC_ACMP3           = 0x03UL    /*!< Analog compare channel 3. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} CCT_ICACMPChannelEnum;

/**
 ******************************************************************************
 * @brief CCT Input capture singal source enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_IC_SINGAL_SOURCE_PIN           = 0x00UL,   /*!< Set input capture singal source as pin. */
  CCT_IC_SINGAL_SOURCE_ACMP          = 0x01UL    /*!< Set input capture singal source as ACMP. */
} CCT_ICSingalSourceEnum;

#if defined (DPM32M08x) || defined (DPM32M05x)
/**
 ******************************************************************************
 * @brief CCT DMA request event enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_DMA_REQ_OVERFLOW          = 0x01UL,   /*!< Timer overflow trigger DMA request. */
  CCT_DMA_REQ_CH0_CAP           = 0x02UL,   /*!< CH0 capture trigger DMA request. */
  CCT_DMA_REQ_CH1_CAP           = 0x03UL,   /*!< CH1 capture trigger DMA request. */
  CCT_DMA_REQ_CH0_CMP           = 0x04UL,   /*!< CH0 compare trigger DMA request. */
  CCT_DMA_REQ_CH1_CMP           = 0x05UL    /*!< CH1 compare trigger DMA request. */
} CCT_DMAReqEnum;
#endif  /* DPM32M08x || DPM32M05x */

/**
 ******************************************************************************
 * @brief CCT channel mode type enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_CH_MODE_CAPTURE          = 0x00UL,   /*!< CCT channel is capture mode. */
  CCT_CH_MODE_COMPARE          = 0x01UL    /*!< CCT channel is compare mode . */
} CCT_ChannelModeEnum;

/**
 ******************************************************************************
 * @brief CCT counter cycle enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_COUNTER_CYCLE_PERIOD           = 0x00UL,              /*!< Period cycle counter. */
  CCT_COUNTER_CYCLE_SINGLE           = CCT_CR_SINGLE_Msk    /*!< Single cycle counter. */
} CCT_CounterCycleEnum;

/**
 ******************************************************************************
 * @brief CCT interrupt type enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_INT_TYPE_OVERFLOW             = CCT_CR_LVAL_IE_Msk,        /*!< Over flow interrupt type. */
  CCT_INT_TYPE_CH0_CAP              = CCT_CR_CH0_CAP_IE_Msk,     /*!< Channel0 capture interrupt type. */
  CCT_INT_TYPE_CH1_CAP              = CCT_CR_CH1_CAP_IE_Msk,     /*!< Channel1 capture interrupt type. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  CCT_INT_TYPE_CH2_CAP              = CCT_CR_CH2_CAP_IE_Msk,     /*!< Channel2 capture interrupt type. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
  CCT_INT_TYPE_CH0_CMP              = CCT_CR_CH0_CMP_IE_Msk,     /*!< Channel0 compare interrupt type. */
  CCT_INT_TYPE_CH1_CMP              = CCT_CR_CH1_CMP_IE_Msk,     /*!< Channel1 compare interrupt type. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  CCT_INT_TYPE_CH2_CMP              = CCT_CR_CH2_CMP_IE_Msk      /*!< Channel1 compare interrupt type. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
} CCT_IntTypeEnum;

/**
 ******************************************************************************
 * @brief CCT interrupt trigger enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_INT_FLAG_OVERFLOW            = CCT_SR_LVAL_IF_Msk,      /*!< Timer overflow trigger interrupt. */
  CCT_INT_FLAG_CH0_CAPTURE         = CCT_SR_CH0_CAP_IF_Msk,   /*!< CH0 capture trigger interrupt. */
  CCT_INT_FLAG_CH1_CAPTURE         = CCT_SR_CH1_CAP_IF_Msk,   /*!< CH1 capture trigger interrupt. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  CCT_INT_FLAG_CH2_CAPTURE         = CCT_SR_CH2_CAP_IF_Msk,   /*!< CH2 capture trigger interrupt. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
  CCT_INT_FLAG_CH0_COMPARE         = CCT_SR_CH0_CMP_IF_Msk,   /*!< CH0 comper trigger interrupt. */
  CCT_INT_FLAG_CH1_COMPARE         = CCT_SR_CH1_CMP_IF_Msk,   /*!< CH1 comper trigger interrupt. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  CCT_INT_FLAG_CH2_COMPARE         = CCT_SR_CH2_CMP_IF_Msk    /*!< CH2 comper trigger interrupt. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
} CCT_IntFlagEnum;

/**
 ******************************************************************************
 * @brief CCT flag enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_FLAG_CH0_RISING_CAPTURE      = CCT_SR_CH0_CAP_RE_Msk,   /*!< CH0 rising capture trigger flag. */
  CCT_FLAG_CH0_FALLING_CAPTURE     = CCT_SR_CH0_CAP_FE_Msk,   /*!< CH0 falling capture trigger flag. */
  CCT_FLAG_CH1_RISING_CAPTURE      = CCT_SR_CH1_CAP_RE_Msk,   /*!< CH1 rising capture trigger flag. */
  CCT_FLAG_CH1_FALLING_CAPTURE     = CCT_SR_CH1_CAP_FE_Msk,   /*!< CH1 falling capture trigger flag. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  CCT_FLAG_CH2_RISING_CAPTURE      = CCT_SR_CH2_CAP_RE_Msk,   /*!< CH2 rising capture trigger flag. */
  CCT_FLAG_CH2_FALLING_CAPTURE     = CCT_SR_CH2_CAP_FE_Msk    /*!< CH2 falling capture trigger flag. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
} CCT_FlagEnum;

/**
 ******************************************************************************
 * @brief CCT IC capture filter sample division value enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_IC_FILT_SAMPLE_DIV1           = 0x00UL,   /*!< Capture filter sample 1 division. */
  CCT_IC_FILT_SAMPLE_DIV4           = 0x01UL,   /*!< Capture filter sample 4 division. */
  CCT_IC_FILT_SAMPLE_DIV16          = 0x02UL,   /*!< Capture filter sample 16 division. */
  CCT_IC_FILT_SAMPLE_DIV32          = 0x03UL    /*!< Capture filter sample 32 division. */
} CCT_ICFiltSampleDivEnum;

/**
 ******************************************************************************
 * @brief CCT IC capture filter length value enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_IC_FILT_LEN1           = 0x00UL,   /*!< Capture filter 1 length. */
  CCT_IC_FILT_LEN8           = 0x01UL,   /*!< Capture filter 8 length. */
  CCT_IC_FILT_LEN16          = 0x02UL,   /*!< Capture filter 16 length. */
  CCT_IC_FILT_LEN32          = 0x03UL    /*!< Capture filter 32 length. */
} CCT_ICFiltLenEnum;

/**
 ******************************************************************************
 * @brief CCT IC after capture action enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_IC_ACTION_NO            = 0x00UL,   /*!< No action after capture. */
  CCT_IC_ACTION_CLEAR_CNT     = 0x01UL    /*!< Clear counter register after capture. */
} CCT_ICActionEnum;

/**
 ******************************************************************************
 * @brief CCT IC capture type enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_IC_POLARITY_NOACTION          = 0x00UL,   /*!< No action. */
  CCT_IC_POLARITY_RISING            = 0x01UL,   /*!< Capture rising. */
  CCT_IC_POLARITY_FALLING           = 0x02UL,   /*!< Capture falling. */
  CCT_IC_POLARITY_RISINGFALLING     = 0x03UL    /*!< Capture rising and falling. */
} CCT_ICPolarityEnum;

/**
 ******************************************************************************
 * @brief CCT clock division value enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_CLK_DIV1           = 0x0000UL,            /*!< Set the clock source of CCTCLK as PCLK. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  CCT_CLK_DIV2           = 0x0100UL,            /*!< Set the clock source of CCTCLK as PCLK/2. */
  CCT_CLK_DIV4           = 0x0200UL,            /*!< Set the clock source of CCTCLK as PCLK/4. */
  CCT_CLK_DIV8           = 0x0300UL,            /*!< Set the clock source of CCTCLK as PCLK/8. */
  CCT_CLK_DIV16          = 0x0400UL,            /*!< Set the clock source of CCTCLK as PCLK/16. */
  CCT_CLK_DIV32          = 0x0500UL,            /*!< Set the clock source of CCTCLK as PCLK/32. */
  CCT_CLK_DIV64          = 0x0600UL,            /*!< Set the clock source of CCTCLK as PCLK/64. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  CCT_CLK_DIV2           = 0x0004UL,            /*!< Set the clock source of CCTCLK as PCLK/2. */
  CCT_CLK_DIV4           = 0x0008UL,            /*!< Set the clock source of CCTCLK as PCLK/4. */
  CCT_CLK_DIV8           = 0x000CUL,            /*!< Set the clock source of CCTCLK as PCLK/8. */
  CCT_CLK_DIV16          = 0x0010UL,            /*!< Set the clock source of CCTCLK as PCLK/16. */
  CCT_CLK_DIV32          = 0x0014UL,            /*!< Set the clock source of CCTCLK as PCLK/32. */
  CCT_CLK_DIV64          = 0x0018UL,            /*!< Set the clock source of CCTCLK as PCLK/64. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
  CCT_CLK_DIV128         = CCT_CR_CLK_DIV_Msk   /*!< Set the clock source of CCTCLK as PCLK/128. */
} CCT_ClockDivEnum;

/**
 ******************************************************************************
 * @brief CCT OC Channel initial output level enumeration.
 *****************************************************************************/
typedef enum
{
  CCT_OC_POLARITY_LOW       = 0x00UL,    /*!< After the counter is reset to zero, the output state is low level. */
  CCT_OC_POLARITY_HIGH      = 0x01UL     /*!< After the counter is reset to zero, the output state is high level. */
} CCT_OCPolarityEnum;


/**
 ******************************************************************************
 * @brief CCT Output Compare initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t                  CCT_OCPulse;          /*!< Specifies the pulse value to be loaded into the Capture Compare Register.
                                                      This parameter can be a number between 0x0000 and 0xFFFF. */
  CCT_OCPolarityEnum        CCT_OCPolarity;       /*!< Specifies the output polarity.
                                                      This parameter can be a value of @ref CCT_OCPolarityEnum. */
} CCT_OCInitTypeStruct;


/**
 ******************************************************************************
 * @brief CCT Input Capture initialization structure definition.
 *****************************************************************************/
typedef struct
{
  CCT_ICSingalSourceEnum    CCT_ICSingalSel;      /*!< Specifies the input singal source.
                                                       This parameter can be a value of @ref CCT_ICSingalSourceEnum. */
  CCT_ICACMPChannelEnum     CCT_ICACMPChl;        /*!< Specifies the ACMP channel.
                                                       This parameter can be a value of @ref CCT_ICACMPChannelEnum. */
  CCT_ICFiltSampleDivEnum   CCT_ICFiltSampDiv;    /*!< Specifies the CCT Input Capture filter sampling and frequency division coefficient.
                                                       This parameter can be a value of @ref CCT_ICFiltSampleDivEnum. */
  CCT_ICFiltLenEnum         CCT_ICFiltLen;        /*!< Specifies the CCT Input Capture filter sampling length.
                                                       This parameter can be a value of @ref CCT_ICFiltLenEnum. */
  CCT_ICActionEnum          CCT_ICAction;         /*!< Specifies the CCT Input Capture action after capture.
                                                       This parameter can be a value of @ref CCT_ICActionEnum. */
  CCT_ICPolarityEnum        CCT_ICPolarity;       /*!< Specifies the input polarity.
                                                       This parameter can be a value of @ref CCT_ICPolarityEnum. */
} CCT_ICInitTypeStruct;

/**
 ******************************************************************************
 * @brief CCT time base initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t                  CCT_ReloadValue;      /*!< Specifies the period value to be loaded into the active Auto-Reload Register at the next update event.
                                                       This parameter can be a number between 0x00000000 and 0xFFFFFFFF.*/
  CCT_ClockDivEnum          CCT_ClockDivision;    /*!< Specifies the clock division.
                                                       This parameter can be a value of @ref CCT_ClockDivEnum */
  CCT_CounterCycleEnum      CCT_CounterCycle;     /*!< Specifies the cycle for the CCTx time base.
                                                       This parameter can be a value of @ref CCT_CounterCycleEnum. */
} CCT_TimeBaseInitTypeStruct;

/* CCT Param Check. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define IS_CCT_PERIPH(x) (((x) == CCT0) || \
                          ((x) == CCT1))
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
#define IS_CCT_PERIPH(x) (((x) == CCT0))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#define IS_CCT_COUNTER_MODE(x) (((x) == CCT_COUNTER_CYCLE_PERIOD) || \
                                ((x) == CCT_COUNTER_CYCLE_SINGLE))

#define IS_CCT_CLOCK_DIV(x) (((x) == CCT_CLK_DIV1) || \
                             ((x) == CCT_CLK_DIV2) || \
                             ((x) == CCT_CLK_DIV4) || \
                             ((x) == CCT_CLK_DIV8) || \
                             ((x) == CCT_CLK_DIV16) || \
                             ((x) == CCT_CLK_DIV32) || \
                             ((x) == CCT_CLK_DIV64) || \
                             ((x) == CCT_CLK_DIV128))

#define IS_CCT_OC_POLARITY(x) (((x) == CCT_OC_POLARITY_HIGH) || \
                               ((x) == CCT_OC_POLARITY_LOW))

#define IS_CCT_IC_SINGLE_SOURCE(x) (((x) == CCT_IC_SINGAL_SOURCE_PIN) || \
                                    ((x) == CCT_IC_SINGAL_SOURCE_ACMP))

#if defined (DPM32M015)
#define IS_CCT_IC_ACMP_CHANNEL(x) (((x) == CCT_IC_ACMP1))
#elif defined (DPM32M036) || defined (DPM32M030)
#define IS_CCT_IC_ACMP_CHANNEL(x) (((x) == CCT_IC_ACMP0) || \
                                   ((x) == CCT_IC_ACMP1))
#else /* (DPM32M08x) || (DPM32M05x) || (DPM32M03x) */
#define IS_CCT_IC_ACMP_CHANNEL(x) (((x) == CCT_IC_ACMP0) || \
                                   ((x) == CCT_IC_ACMP1) || \
                                   ((x) == CCT_IC_ACMP2) || \
                                   ((x) == CCT_IC_ACMP3))
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
   
#define IS_CCT_IC_FILT_SAMPLE_DIV(x) (((x) == CCT_IC_FILT_SAMPLE_DIV1) || \
                                      ((x) == CCT_IC_FILT_SAMPLE_DIV4) || \
                                      ((x) == CCT_IC_FILT_SAMPLE_DIV16) || \
                                      ((x) == CCT_IC_FILT_SAMPLE_DIV32))

#define IS_CCT_IC_FILT_LEN(x) (((x) == CCT_IC_FILT_LEN1) || \
                               ((x) == CCT_IC_FILT_LEN8) || \
                               ((x) == CCT_IC_FILT_LEN16) || \
                               ((x) == CCT_IC_FILT_LEN32))

#define IS_CCT_IC_ACTION(x) (((x) == CCT_IC_ACTION_CLEAR_CNT) || \
                             ((x) == CCT_IC_ACTION_NO))

#define IS_CCT_IC_POLARITY(x) (((x) == CCT_IC_POLARITY_RISING) || \
                               ((x) == CCT_IC_POLARITY_FALLING) || \
                               ((x) == CCT_IC_POLARITY_RISINGFALLING))

#if defined (DPM32M08x) || defined (DPM32M05x)
#define IS_CCT_DMA_REQ(x) (((x) == CCT_DMA_REQ_OVERFLOW) || \
                           ((x) == CCT_DMA_REQ_CH0_CAP) || \
                           ((x) == CCT_DMA_REQ_CH1_CAP) || \
                           ((x) == CCT_DMA_REQ_CH0_CMP) || \
                           ((x) == CCT_DMA_REQ_CH1_CMP))
#endif  /* DPM32M08x || DPM32M05x */

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) 

#define IS_CCT_TNT_TYPE(x) (((x & CCT_INT_TYPE_OVERFLOW) == CCT_INT_TYPE_OVERFLOW) || \
                            ((x & CCT_INT_TYPE_CH0_CAP) == CCT_INT_TYPE_CH0_CAP) || \
                            ((x & CCT_INT_TYPE_CH1_CAP) == CCT_INT_TYPE_CH1_CAP) || \
                            ((x & CCT_INT_TYPE_CH0_CMP) == CCT_INT_TYPE_CH0_CMP) || \
                            ((x & CCT_INT_TYPE_CH1_CMP) == CCT_INT_TYPE_CH1_CMP))

#define IS_CCT_TNT_FLAG(x) (((x) == CCT_INT_FLAG_OVERFLOW) || \
                            ((x) == CCT_INT_FLAG_CH0_CAPTURE) || \
                            ((x) == CCT_INT_FLAG_CH1_CAPTURE) || \
                            ((x) == CCT_INT_FLAG_CH0_COMPARE) || \
                            ((x) == CCT_INT_FLAG_CH1_COMPARE))

#define IS_CCT_FLAG(x) (((x) == CCT_FLAG_CH0_RISING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH0_FALLING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH1_RISING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH1_FALLING_CAPTURE))

#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)

#define IS_CCT_TNT_TYPE(x) (((x & CCT_INT_TYPE_OVERFLOW) == CCT_INT_TYPE_OVERFLOW) || \
                            ((x & CCT_INT_TYPE_CH0_CAP) == CCT_INT_TYPE_CH0_CAP) || \
                            ((x & CCT_INT_TYPE_CH1_CAP) == CCT_INT_TYPE_CH1_CAP) || \
                            ((x & CCT_INT_TYPE_CH2_CAP) == CCT_INT_TYPE_CH2_CAP) || \
                            ((x & CCT_INT_TYPE_CH0_CMP) == CCT_INT_TYPE_CH0_CMP) || \
                            ((x & CCT_INT_TYPE_CH1_CMP) == CCT_INT_TYPE_CH1_CMP) || \
                            ((x & CCT_INT_TYPE_CH2_CMP) == CCT_INT_TYPE_CH2_CMP))

#define IS_CCT_TNT_FLAG(x) (((x) == CCT_INT_FLAG_OVERFLOW) || \
                            ((x) == CCT_INT_FLAG_CH0_CAPTURE) || \
                            ((x) == CCT_INT_FLAG_CH1_CAPTURE) || \
                            ((x) == CCT_INT_FLAG_CH2_CAPTURE) || \
                            ((x) == CCT_INT_FLAG_CH0_COMPARE) || \
                            ((x) == CCT_INT_FLAG_CH1_COMPARE) || \
                            ((x) == CCT_INT_FLAG_CH2_COMPARE))

#define IS_CCT_FLAG(x) (((x) == CCT_FLAG_CH0_RISING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH0_FALLING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH1_RISING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH1_FALLING_CAPTURE)|| \
                        ((x) == CCT_FLAG_CH2_RISING_CAPTURE) || \
                        ((x) == CCT_FLAG_CH2_FALLING_CAPTURE))

#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/* CCT time base functions. */
void                    CCT_DeInit(CCT_Type* CCTx);
void                    CCT_TimeBaseInit(CCT_Type* CCTx, CCT_TimeBaseInitTypeStruct* CCT_TimeBaseInitType);
void                    CCT_TimeBaseStructInit(CCT_TimeBaseInitTypeStruct* CCT_TimeBaseInitType);
void                    CCT_Cmd(CCT_Type* CCTx, FunctionalState NewState);
void                    CCT_SetAutoReload(CCT_Type* CCTx, uint16_t CCT_Value);
uint16_t                CCT_GetAutoReload(CCT_Type* CCTx);
uint16_t                CCT_GetCounter(CCT_Type* CCTx);
void                    CCT_CounterCycleConfig(CCT_Type* CCTx, CCT_CounterCycleEnum CCT_Cycle);
void                    CCT_ClockDivConfig(CCT_Type* CCTx, CCT_ClockDivEnum CCT_Div);

/* CCT output compare functions. */
void                    CCT_OC0Init(CCT_Type* CCTx, CCT_OCInitTypeStruct* CCT_OCInitType);
void                    CCT_OC1Init(CCT_Type* CCTx, CCT_OCInitTypeStruct* CCT_OCInitType);
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    CCT_OC2Init(CCT_Type *CCTx, CCT_OCInitTypeStruct *CCT_OCInitType);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

void                    CCT_OCStructInit(CCT_OCInitTypeStruct* CCT_OCInitType);
void                    CCT_SetCompare0(CCT_Type* CCTx, uint16_t CCT_Value);
void                    CCT_SetCompare1(CCT_Type* CCTx, uint16_t CCT_Value);
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    CCT_SetCompare2(CCT_Type *CCTx, uint16_t CCT_Value);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
void                    CCT_OC0PolarityConfig(CCT_Type* CCTx, CCT_OCPolarityEnum CCT_Polarity);
void                    CCT_OC1PolarityConfig(CCT_Type* CCTx, CCT_OCPolarityEnum CCT_Polarity);
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    CCT_OC2PolarityConfig(CCT_Type* CCTx, CCT_OCPolarityEnum CCT_Polarity);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/* CCT input capture functions. */
void                    CCT_ICStructInit(CCT_ICInitTypeStruct* CCT_ICInitType);
FlagState               CCT_GetFlagStatus(CCT_Type* CCTx, CCT_FlagEnum CCT_Flag);
void                    CCT_ClearFlag(CCT_Type* CCTx, CCT_FlagEnum CCT_Flag);

void                    CCT_IC0Init(CCT_Type* CCTx, CCT_ICInitTypeStruct* CCT_ICInitType);
void                    CCT_IC0SingleSourceConfig(CCT_Type* CCTx, CCT_ICSingalSourceEnum CCT_Source);
void                    CCT_IC0ACMPChannelConfig(CCT_Type* CCTx, CCT_ICACMPChannelEnum CCT_Chl);
void                    CCT_IC0FltSampDivConfig(CCT_Type* CCTx, CCT_ICFiltSampleDivEnum CCT_Div);
void                    CCT_IC0FltLenConfig(CCT_Type* CCTx, CCT_ICFiltLenEnum CCT_Len);
void                    CCT_IC0ActionConfig(CCT_Type* CCTx, CCT_ICActionEnum CCT_Act);
void                    CCT_IC0PolarityConfig(CCT_Type* CCTx, CCT_ICPolarityEnum CCT_Pol);
uint16_t                CCT_GetCapture0(CCT_Type* CCTx);

void                    CCT_IC1Init(CCT_Type* CCTx, CCT_ICInitTypeStruct* CCT_ICInitType);
void                    CCT_IC1SingleSourceConfig(CCT_Type* CCTx, CCT_ICSingalSourceEnum CCT_Source);
void                    CCT_IC1ACMPChannelConfig(CCT_Type* CCTx, CCT_ICACMPChannelEnum CCT_Chl);
void                    CCT_IC1FltSampDivConfig(CCT_Type* CCTx, CCT_ICFiltSampleDivEnum CCT_Div);
void                    CCT_IC1FltLenConfig(CCT_Type* CCTx, CCT_ICFiltLenEnum CCT_Len);
void                    CCT_IC1ActionConfig(CCT_Type* CCTx, CCT_ICActionEnum CCT_Act);
void                    CCT_IC1PolarityConfig(CCT_Type* CCTx, CCT_ICPolarityEnum CCT_Pol);
uint16_t                CCT_GetCapture1(CCT_Type* CCTx);

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    CCT_IC2Init(CCT_Type* CCTx, CCT_ICInitTypeStruct* CCT_ICInitType);
void                    CCT_IC2SingleSourceConfig(CCT_Type* CCTx, CCT_ICSingalSourceEnum CCT_Source);
void                    CCT_IC2ACMPChannelConfig(CCT_Type* CCTx, CCT_ICACMPChannelEnum CCT_Chl);
void                    CCT_IC2FltSampDivConfig(CCT_Type* CCTx, CCT_ICFiltSampleDivEnum CCT_Div);
void                    CCT_IC2FltLenConfig(CCT_Type* CCTx, CCT_ICFiltLenEnum CCT_Len);
void                    CCT_IC2ActionConfig(CCT_Type* CCTx, CCT_ICActionEnum CCT_Act);
void                    CCT_IC2PolarityConfig(CCT_Type* CCTx, CCT_ICPolarityEnum CCT_Pol);
uint16_t                CCT_GetCapture2(CCT_Type* CCTx);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/* CCT channel enable or disable functions. */
void                    CCT_CH0Cmd(CCT_Type* CCTx, FunctionalState NewState);
void                    CCT_CH1Cmd(CCT_Type* CCTx, FunctionalState NewState);
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    CCT_CH2Cmd(CCT_Type* CCTx, FunctionalState NewState);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#if defined (DPM32M08x) || defined (DPM32M05x)
/* CCT DMA functions. */
void                    CCT_DMACmd(CCT_Type* CCTx, CCT_DMAReqEnum CCT_DMAReq, FunctionalState NewState);
#endif  /* DPM32M08x || DPM32M05x */

/* CCT interrupt functions. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    CCT_IntCmd(CCT_Type* CCTx, uint32_t CCT_IntType, FunctionalState NewState);
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
void                    CCT_IntCmd(CCT_Type* CCTx, uint16_t CCT_IntType, FunctionalState NewState);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
FunctionalState         CCT_GetIntCmdStatus(CCT_Type* CCTx, CCT_IntTypeEnum CCT_IntType);
FlagState               CCT_GetIntFlagStatus(CCT_Type* CCTx, CCT_IntFlagEnum CCT_IntFlag);
void                    CCT_ClearIntFlag(CCT_Type* CCTx, CCT_IntFlagEnum CCT_IntFlag);

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_CCT_H__ */
