/**
 *******************************************************************************
 * @file    dpm32m0xx_epwm.h
 *
 * @brief   Header file for EPWM firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_EPWM_H__
#define __DPM32M0XX_EPWM_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"

#define EPWM_CHCMP1(channel)            REG32((EPWM_BASE + 0x1CUL) + 0x08UL*(channel))   /*!< the address of EPWM channel compare value1 register */
#define EPWM_CHCMP2(channel)            REG32((EPWM_BASE + 0x20UL) + 0x08UL*(channel))   /*!< the address of EPWM channel compare value2 register */

/**
 ******************************************************************************
 * @brief EPWM channel enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CH0           = 0x00UL,   /*!< EPWM channel 0. */
  EPWM_CH1           = 0x01UL,   /*!< EPWM channel 1. */
  EPWM_CH2           = 0x02UL,   /*!< EPWM channel 2. */

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  EPWM_CH3           = 0x03UL    /*!< EPWM channel 3. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} EPWM_ChannelEnum;

/**
 ******************************************************************************
 * @brief EPWM clock division value enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CLK_DIV1           = 0x00UL,  /*!< Set the clock source of EPWMCLK as PCLK. */
  EPWM_CLK_DIV2           = 0x01UL,  /*!< Set the clock source of EPWMCLK as PCLK/2. */
  EPWM_CLK_DIV4           = 0x02UL,  /*!< Set the clock source of EPWMCLK as PCLK/4. */
  EPWM_CLK_DIV8           = 0x03UL,  /*!< Set the clock source of EPWMCLK as PCLK/8. */
  EPWM_CLK_DIV16          = 0x04UL,  /*!< Set the clock source of EPWMCLK as PCLK/16. */
  EPWM_CLK_DIV32          = 0x05UL,  /*!< Set the clock source of EPWMCLK as PCLK/32. */
  EPWM_CLK_DIV64          = 0x06UL,  /*!< Set the clock source of EPWMCLK as PCLK/64. */
  EPWM_CLK_DIV128         = 0x07UL   /*!< Set the clock source of EPWMCLK as PCLK/128. */
} EPWM_ClockDivEnum;

/**
 ******************************************************************************
 * @brief EPWM counter mode enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_COUNTER_CYCLE_PERIOD           = 0x00UL,   /*!< Period cycle counter. */
  EPWM_COUNTER_CYCLE_SINGLE           = 0x01UL    /*!< Single cycle counter. */
} EPWM_CounterCycleEnum;

/**
 ******************************************************************************
 * @brief EPWM counter mode enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_COUNTER_MODE_INCREASE_DECREASE     = 0x00UL,   /*!< Increase and decrease counter mode. */
  EPWM_COUNTER_MODE_INCREASE              = 0x01UL    /*!< Increase counter mode. */
} EPWM_CounterModeEnum;

/**
 ******************************************************************************
 * @brief EPWM interrupt type enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_INT_TYPE_LOAD_VAL          = EPWM_CR_LOAD_VAL_IE_Msk,  /*!< Counter equal to load value interrupt. */
  EPWM_INT_TYPE_PERIOD            = EPWM_CR_PERIOD_IE_Msk,    /*!< Counter equal to period interrupt. */
  EPWM_INT_TYPE_ADC_CMP1          = EPWM_CR_ADC_CMP1_IE_Msk,  /*!< Counter equal to ADC_CMP1 value interrupt. */
  EPWM_INT_TYPE_ADC_CMP2          = EPWM_CR_ADC_CMP2_IE_Msk,  /*!< Counter equal to ADC_CMP2 value interrupt. */
  EPWM_INT_TYPE_STOP              = EPWM_CR_STOP_IE_Msk       /*!< Stop interrupt. */
} EPWM_IntTypeEnum;

/**
 ******************************************************************************
 * @brief EPWM interrupt flag enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_INT_FLAG_LOAD_VAL          = EPWM_SR_LOAD_VAL_IF_Msk,  /*!< Counter equal to load value interrupt flag. */
  EPWM_INT_FLAG_PERIOD            = EPWM_SR_PERIOD_IF_Msk,    /*!< Counter equal to period interrupt flag. */
  EPWM_INT_FLAG_ADC_CMP1          = EPWM_SR_ADC_CMP1_IF_Msk,  /*!< Counter equal to ADC_CMP1 value interrupt flag. */
  EPWM_INT_FLAG_ADC_CMP2          = EPWM_SR_ADC_CMP2_IF_Msk,  /*!< Counter equal to ADC_CMP2 value interrupt flag. */
  EPWM_INT_FLAG_STOP              = EPWM_SR_STOP_IF_Msk       /*!< Stop interrupt flag. */
} EPWM_IntFlagEnum;

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
/**
 ******************************************************************************
 * @brief EPWM compare valid direction enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CMP_DIR_INCREASE            = 0x00UL,   /*!< Counter increase takes valid . */
  EPWM_CMP_DIR_DECREASE            = 0x01UL    /*!< Counter decrease takes valid . */
} EPWM_CmpDirEnum;
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
/**
 ******************************************************************************
 * @brief EPWM compare valid direction enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CMP_DIR_DISABLE             = 0x00UL,    /*!< Counter is not valid . */
  EPWM_CMP_DIR_INCREASE            = 0x01UL,    /*!< Counter increase takes valid . */
  EPWM_CMP_DIR_DECREASE            = 0x02UL,    /*!< Counter decrease takes valid . */
  EPWM_CMP_DIR_INCREASE_DECREASE   = 0x03UL     /*!< Counter increase and decrease takes valid . */
} EPWM_CmpDirEnum;
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */


/**
 ******************************************************************************
 * @brief EPWM compare output action enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CMP_ACT_OUTPUT_LOW          = 0x00UL,   /*!< The count value is equal to the comparison value output 0 . */
  EPWM_CMP_ACT_OUTPUT_HIGH         = 0x01UL,   /*!< The count value is equal to the comparison value output 1 . */
  EPWM_CMP_ACT_OUTPUT_FLIP         = 0x02UL,   /*!< The count value is equal to the comparison value output flip . */
  EPWM_CMP_ACT_OUTPUT_UNCHANGE     = 0x03UL    /*!< The count value is equal to the comparison value output unchange . */
} EPWM_CmpActEnum;

/**
 ******************************************************************************
 * @brief EPWM channel in initial output status enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CH_INIT_STATUS_0            = 0x00UL,   /*!< Channel in initial output level is 0 status. */
  EPWM_CH_INIT_STATUS_1            = 0x01UL    /*!< Channel in initial output level is 1 status. */
} EPWM_CHInitStatusEnum;

/**
 ******************************************************************************
 * @brief EPWM channel output polarity enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_CH_OUT_POLARITY_NORMAL        = 0x00UL,   /*!< Channel normal output. */
  EPWM_CH_OUT_POLARITY_INVERT        = 0x01UL    /*!< Channel invert output. */
} EPWM_CHOutPolarityEnum;

/**
 ******************************************************************************
 * @brief EPWM force output enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_FO_LEVEL_LOW        = 0x00UL,   /*!< Force output low level. */
  EPWM_FO_LEVEL_HIGH       = 0x01UL    /*!< Force output high level. */
} EPWM_FOLevelEnum;

/**
 ******************************************************************************
 * @brief EPWM emergency stop filter sample division enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_FILT_SAMPLE_DIV1        = 0x00UL,                        /*!< The emergency stop filter sampling clock is equal to PCLK . */
  EPWM_ES_FILT_SAMPLE_DIV4        = 0x01UL,                        /*!< The emergency stop filter sampling clock is equal to PCLK/4. */
  EPWM_ES_FILT_SAMPLE_DIV16       = 0x02UL,                        /*!< The emergency stop filter sampling clock is equal to PCLK/16. */
  EPWM_ES_FILT_SAMPLE_DIV32       = EPWM_STOP_CR_FLT_SAMPLE_Msk    /*!< The emergency stop filter sampling clock is equal to PCLK/32. */
} EPWM_ESFiltSampleDivEnum;

/**
 ******************************************************************************
 * @brief EPWM emergency stop filter cycles length enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_FLT_LEN1        = 0x00UL,   /*!< Filter samples 1 cycles length. */
  EPWM_ES_FLT_LEN8        = 0x01UL,   /*!< Filter samples 8 cycles length. */
  EPWM_ES_FLT_LEN16       = 0x02UL,   /*!< Filter samples 16 cycles length. */
  EPWM_ES_FLT_LEN32       = 0x03UL    /*!< Filter samples 32 cycles length. */
} EPWM_ESFiltLenEnum;

/**
 ******************************************************************************
 * @brief EPWM emergency stop external trigger polarity enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_EXT_POLARITY_LOW        = 0x00UL,   /*!< Emergency stop external trigger valid level is low level. */
  EPWM_ES_EXT_POLARITY_HIGH       = 0x01UL    /*!< Emergency stop external trigger valid level is high level. */
} EPWM_ESExtPolarityEnum;

/**
 ******************************************************************************
 * @brief EPWM emergency stop ACMP trigger polarity enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_ACMP_POLARITY_LOW        = 0x00UL,   /*!< Emergency stop ACMP trigger valid level is low level. */
  EPWM_ES_ACMP_POLARITY_HIGH       = 0x01UL    /*!< Emergency stop ACMP trigger valid level is high level. */
} EPWM_ESACMPPolarityEnum;

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
/**
 ******************************************************************************
 * @brief EPWM emergency stop ACMP mode enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_ACMP_MODE_LATCH        = 0x00UL,     /*!< Emergency stop ACMP with latch mode. */
  EPWM_ES_ACMP_MODE_WAVE         = 0x01UL      /*!< Emergency stop ACMP with wave mode. */
} EPWM_ESACMPModeEnum;
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 ******************************************************************************
 * @brief EPWM emergency stop trigger source ACMP channel enumeration.
 *****************************************************************************/
typedef enum
{
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036) || defined (DPM32M030)
  EPWM_ES_ACMP0       = 0x00UL,                 /*!< Emergency stop trigger source select ACMP0. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M015 */
  EPWM_ES_ACMP1       = 0x01UL,                 /*!< Emergency stop trigger source select ACMP1. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  EPWM_ES_ACMP2       = 0x02UL,                 /*!< Emergency stop trigger source select ACMP2. */
  EPWM_ES_ACMP3       = 0x03UL                  /*!< Emergency stop trigger source select ACMP3. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} EPWM_ESACMPChannelEnum;

/**
 ******************************************************************************
 * @brief EPWM emergency stop channel output level enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_CH_OUT_LEVEL_LOW       = 0x00UL,   /*!< Emergency stop channel output low level. */
  EPWM_ES_CH_OUT_LEVEL_HIGH      = 0x01UL    /*!< Emergency stop channel output high level. */
} EPWM_ESOutLevelEnum;

/**
 ******************************************************************************
 * @brief EPWM emergency stop status enumeration.
 *****************************************************************************/
typedef enum
{
  EPWM_ES_STOP_STATUS_SOFT      = EPWM_STOP_SR_STOP_Msk,       /*!< Software emergency stop. */
  EPWM_ES_STOP_STATUS_EXT       = EPWM_STOP_SR_EXT_TRIG_Msk,   /*!< External emergency stop. */
  EPWM_ES_STOP_STATUS_ACMP      = EPWM_STOP_SR_ACMP_TRIG_Msk   /*!< ACMP emergency stop. */
} EPWM_ESStopStatusEnum;

/**
 ******************************************************************************
 * @brief EPWM time base initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t                  EPWM_ReloadValue;       /*!< Specifies the period value to be loaded into the active Auto-Reload Register at the next update event.
                                                         This parameter can be a number between 0x0000 and 0xFFFF.*/
  EPWM_ClockDivEnum         EPWM_ClockDiv;          /*!< Specifies the clock division.
                                                         This parameter can be a value of @ref EPWM_ClockDivEnum */
  EPWM_CounterCycleEnum     EPWM_CounterCycle;      /*!< Specifies the cycle for the EPWM time base.
                                                         This parameter can be a value of @ref EPWM_CounterCycleEnum. */
  EPWM_CounterModeEnum      EPWM_CounterMode;       /*!< Specifies the mode for the EPWM time base.
                                                         This parameter can be a value of @ref EPWM_CounterModeEnum. */
} EPWM_TimeBaseInitTypeStruct;

/**
 ******************************************************************************
 * @brief EPWM ADC compare initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t                  EPWM_ADCCmp1Value;         /*!< Specifies the ADC cpomare1 value.
                                                            This parameter can be a number between 0x0000 and 0xFFFF.*/
  uint16_t                  EPWM_ADCCmp2Value;         /*!< Specifies the ADC cpomare1 value.
                                                            This parameter can be a number between 0x0000 and 0xFFFF.*/
  EPWM_CmpDirEnum           EPWM_ADCCmp1Dir;           /*!< ADC compare vaild direction.
                                                            This parameter can be a value of @ref EPWM_CmpDirEnum.*/
  EPWM_CmpDirEnum           EPWM_ADCCmp2Dir;           /*!< ADC compare vaild direction.
                                                            This parameter can be a value of @ref EPWM_CmpDirEnum.*/

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  FunctionalState           EPWM_ADCCmp1State;         /*!< Specifies the ADC compare state.
                                                            This parameter can be a value of @ref FunctionalState. */
  FunctionalState           EPWM_ADCCmp2State;         /*!< Specifies the ADC compare state.
                                                            This parameter can be a value of @ref FunctionalState. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} EPWM_ADCCmpInitTypeStruct;

/**
 ******************************************************************************
 * @brief EPWM channel initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t                  EPWM_Compare1Value;         /*!< Specifies the compare1 value.
                                                             This parameter can be a number between 0x0000 and 0xFFFF.*/
  uint16_t                  EPWM_Compare2Value;         /*!< Specifies the compare2 value.
                                                             This parameter can be a number between 0x0000 and 0xFFFF.*/
  EPWM_CmpActEnum           EPWM_Compare1Action;        /*!< Specifies the compare action.
                                                             This parameter can be a value of @ref EPWM_CmpActEnum. */
  EPWM_CmpActEnum           EPWM_Compare2Action;        /*!< Specifies the compare action.
                                                             This parameter can be a value of @ref EPWM_CmpActEnum. */
  EPWM_CHInitStatusEnum     EPWM_CHInitStatus;          /*!< Specifies the initial output status.
                                                             This parameter can be a value of @ref EPWM_CHInitStatusEnum. */
  EPWM_CHOutPolarityEnum    EPWM_NPolarity;             /*!< Specifies the N channel output state.
                                                             This parameter can be a value of @ref EPWM_CHOutPolarityEnum. */
  EPWM_CHOutPolarityEnum    EPWM_PPolarity;             /*!< Specifies the P channel output state.
                                                             This parameter can be a value of @ref EPWM_CHOutPolarityEnum. */
  EPWM_CmpDirEnum           EPWM_Compare1Dir;           /*!< Specifies the compare1 vaild direction.
                                                             This parameter can be a value of @ref EPWM_CmpDirEnum. */
  EPWM_CmpDirEnum           EPWM_Compare2Dir;           /*!< Specifies the compare2 vaild direction.
                                                             This parameter can be a value of @ref EPWM_CmpDirEnum. */

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  FunctionalState           EPWM_Compare1State;         /*!< Specifies the compare1 state.
                                                             This parameter can be a value of @ref FunctionalState. */
  FunctionalState           EPWM_Compare2State;         /*!< Specifies the compare2 state.
                                                             This parameter can be a value of @ref FunctionalState. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} EPWM_CHInitTypeStruct;


/**
 ******************************************************************************
 * @brief EPWM channel force output initialization structure definition.
 *****************************************************************************/
typedef struct
{
  EPWM_FOLevelEnum          EPWM_FONLevel;         /*!< Specifies the N channel force output level.
                                                        This parameter can be a value of @ref EPWM_FOLevelEnum. */
  EPWM_FOLevelEnum          EPWM_FOPLevel;         /*!< Specifies the P channel force output level.
                                                        This parameter can be a value of @ref EPWM_FOLevelEnum. */
  FunctionalState           EPWM_FONState;         /*!< Specifies the N channel force output state.
                                                        This parameter can be a value of @ref FunctionalState. */
  FunctionalState           EPWM_FOPState;         /*!< Specifies the P channel force output state.
                                                        This parameter can be a value of @ref FunctionalState. */
} EPWM_FOInitTypeStruct;

/**
 ******************************************************************************
 * @brief EPWM emergency stop initialization structure definition.
 *****************************************************************************/
typedef struct
{
  EPWM_ESFiltSampleDivEnum      EPWM_ESFiltSampleDiv;         /*!< Specifies emergency stop filter sample division.
                                                                   This parameter can be a value of @ref EPWM_ESFiltSampleDivEnum. */
  EPWM_ESFiltLenEnum            EPWM_ESFiltLen;               /*!< Specifies emergency stop filter length.
                                                                   This parameter can be a value of @ref EPWM_ESFiltLenEnum. */
  EPWM_ESExtPolarityEnum        EPWM_ESExtPolarity;           /*!< Specifies emergency stop external trigger polarity.
                                                                   This parameter can be a value of @ref EPWM_ESExtPolarityEnum. */
  FunctionalState               EPWM_ESExtState;              /*!< Specifies emergency stop external trigger state.
                                                                   This parameter can be a value of @ref FunctionalState. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
  EPWM_ESACMPModeEnum           EPWM_ESACMPMode;              /*!< Specifies emergency stop ACMP mode.
                                                                   This parameter can be a value of @ref EPWM_ESACMPModeEnum. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

  EPWM_ESACMPPolarityEnum       EPWM_ESACMPPolarity;          /*!< Specifies emergency stop ACMP trigger polarity.
                                                                   This parameter can be a value of @ref EPWM_ESACMPPolarityEnum. */
  EPWM_ESACMPChannelEnum        EPWM_ESACMPChannel;           /*!< Specifies emergency stop ACMP channel.
                                                                   This parameter can be a value of @ref EPWM_ESACMPChannelEnum. */
  FunctionalState               EPWM_ESACMPState;             /*!< Specifies emergency stop ACMP trigger state.
                                                                   This parameter can be a value of @ref FunctionalState. */
  FunctionalState               EPWM_ESDebugState;            /*!< Specifies emergency stop debug mode switch to stop level.
                                                                   This parameter can be a value of @ref FunctionalState. */
  EPWM_ESOutLevelEnum           EPWM_ESCH0NLevel;             /*!< Specifies emergency stop channel0 Nchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
  EPWM_ESOutLevelEnum           EPWM_ESCH0PLevel;             /*!< Specifies emergency stop channel0 Pchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
  EPWM_ESOutLevelEnum           EPWM_ESCH1NLevel;             /*!< Specifies emergency stop channel1 Nchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
  EPWM_ESOutLevelEnum           EPWM_ESCH1PLevel;             /*!< Specifies emergency stop channel1 Pchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
  EPWM_ESOutLevelEnum           EPWM_ESCH2NLevel;             /*!< Specifies emergency stop channel2 Nchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
  EPWM_ESOutLevelEnum           EPWM_ESCH2PLevel;             /*!< Specifies emergency stop channel2 Pchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  EPWM_ESOutLevelEnum           EPWM_ESCH3NLevel;             /*!< Specifies emergency stop channel3 Nchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
  EPWM_ESOutLevelEnum           EPWM_ESCH3PLevel;             /*!< Specifies emergency stop channel3 Pchannel output level.
                                                                   This parameter can be a value of @ref EPWM_ESOutLevelEnum. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} EPWM_ESInitTypeStruct;


/* EPWM Param Check. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define IS_EPWM_CHANNEL(x) (((x) == EPWM_CH0) || \
                            ((x) == EPWM_CH1) || \
                            ((x) == EPWM_CH2) || \
                            ((x) == EPWM_CH3))
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_EPWM_CHANNEL(x) (((x) == EPWM_CH0) || \
                            ((x) == EPWM_CH1) || \
                            ((x) == EPWM_CH2))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */


#define IS_EPWM_CLOCK_DIV(x) (((x) == EPWM_CLK_DIV1) || \
                              ((x) == EPWM_CLK_DIV2) || \
                              ((x) == EPWM_CLK_DIV4) || \
                              ((x) == EPWM_CLK_DIV8) || \
                              ((x) == EPWM_CLK_DIV16) || \
                              ((x) == EPWM_CLK_DIV32) || \
                              ((x) == EPWM_CLK_DIV64) || \
                              ((x) == EPWM_CLK_DIV128))

#define IS_EPWM_COUNTER_CYCLE(x) (((x) == EPWM_COUNTER_CYCLE_PERIOD) || \
                                  ((x) == EPWM_COUNTER_CYCLE_SINGLE))

#define IS_EPWM_COUNTER_MODE(x) (((x) == EPWM_COUNTER_MODE_INCREASE_DECREASE) || \
                                 ((x) == EPWM_COUNTER_MODE_INCREASE))

#define IS_EPWM_INT_TYPE(x) (((x & EPWM_INT_TYPE_LOAD_VAL) == EPWM_INT_TYPE_LOAD_VAL) || \
                             ((x & EPWM_INT_TYPE_PERIOD) == EPWM_INT_TYPE_PERIOD) || \
                             ((x & EPWM_INT_TYPE_ADC_CMP1) == EPWM_INT_TYPE_ADC_CMP1) || \
                             ((x & EPWM_INT_TYPE_ADC_CMP2) == EPWM_INT_TYPE_ADC_CMP2) || \
                             ((x & EPWM_INT_TYPE_STOP) == EPWM_INT_TYPE_STOP))


#define IS_EPWM_INT_FLAG(x) (((x) == EPWM_INT_FLAG_LOAD_VAL) || \
                             ((x) == EPWM_INT_FLAG_PERIOD) || \
                             ((x) == EPWM_INT_FLAG_ADC_CMP1) || \
                             ((x) == EPWM_INT_FLAG_ADC_CMP2) || \
                             ((x) == EPWM_INT_FLAG_STOP))

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define IS_EPWM_CMP_DIRECTION(x) (((x) == EPWM_CMP_DIR_INCREASE) || \
                                  ((x) == EPWM_CMP_DIR_DECREASE))
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_EPWM_CMP_DIRECTION(x) (((x) == EPWM_CMP_DIR_DISABLE) || \
                                  ((x) == EPWM_CMP_DIR_INCREASE) || \
                                  ((x) == EPWM_CMP_DIR_DECREASE) || \
                                  ((x) == EPWM_CMP_DIR_INCREASE_DECREASE))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#define IS_EPWM_CMP_ACTION(x) (((x) == EPWM_CMP_ACT_OUTPUT_LOW) || \
                               ((x) == EPWM_CMP_ACT_OUTPUT_HIGH) || \
                               ((x) == EPWM_CMP_ACT_OUTPUT_FLIP) || \
                               ((x) == EPWM_CMP_ACT_OUTPUT_UNCHANGE))

#define IS_EPWM_CH_INIT_STATUS(x) (((x) == EPWM_CH_INIT_STATUS_0) || \
                                   ((x) == EPWM_CH_INIT_STATUS_1))

#define IS_EPWM_CH_OUT_POLARITY(x) (((x) == EPWM_CH_OUT_POLARITY_NORMAL) || \
                                    ((x) == EPWM_CH_OUT_POLARITY_INVERT))

#define IS_EPWM_FORCE_OUT_LEVEL(x) (((x) == EPWM_FO_LEVEL_LOW) || \
                                    ((x) == EPWM_FO_LEVEL_HIGH))

#define IS_EPWM_DEAD_TIME_LEN(x) ((x) <= 0x3FF)

#define IS_EPWM_ES_FILT_SAMPEL_DIV(x) (((x) == EPWM_ES_FILT_SAMPLE_DIV1) || \
                                       ((x) == EPWM_ES_FILT_SAMPLE_DIV4) || \
                                       ((x) == EPWM_ES_FILT_SAMPLE_DIV16) || \
                                       ((x) == EPWM_ES_FILT_SAMPLE_DIV32))

#define IS_EPWM_ES_FILT_LEN(x) (((x) == EPWM_ES_FLT_LEN1) || \
                                ((x) == EPWM_ES_FLT_LEN8) || \
                                ((x) == EPWM_ES_FLT_LEN16) || \
                                ((x) == EPWM_ES_FLT_LEN32))

#define IS_EPWM_ES_EXT_POLARITY(x) (((x) == EPWM_ES_EXT_POLARITY_LOW) || \
                                    ((x) == EPWM_ES_EXT_POLARITY_HIGH))

#define IS_EPWM_ES_ACMP_POLARITY(x) (((x) == EPWM_ES_ACMP_POLARITY_LOW) || \
                                     ((x) == EPWM_ES_ACMP_POLARITY_HIGH))

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define IS_EPWM_ES_ACMP_CHANNEL(x) (((x) == EPWM_ES_ACMP0) || \
                                    ((x) == EPWM_ES_ACMP1) || \
                                    ((x) == EPWM_ES_ACMP2) || \
                                    ((x) == EPWM_ES_ACMP3))
#elif defined (DPM32M036) || defined (DPM32M030)
#define IS_EPWM_ES_ACMP_CHANNEL(x) (((x) == EPWM_ES_ACMP0) || \
                                    ((x) == EPWM_ES_ACMP1))
#elif defined (DPM32M015)
#define IS_EPWM_ES_ACMP_CHANNEL(x) (((x) == EPWM_ES_ACMP1))
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_EPWM_ES_ACMP_MODE(x) (((x) == EPWM_ES_ACMP_MODE_LATCH) || \
                                    ((x) == EPWM_ES_ACMP_MODE_WAVE))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#define IS_EPWM_ES_CH_OUT_LEVEL(x) (((x) == EPWM_ES_CH_OUT_LEVEL_LOW) || \
                                    ((x) == EPWM_ES_CH_OUT_LEVEL_HIGH))

#define IS_EPWM_ES_STOP_STATUS(x) (((x) == EPWM_ES_STOP_STATUS_SOFT) || \
                                   ((x) == EPWM_ES_STOP_STATUS_EXT) || \
                                   ((x) == EPWM_ES_STOP_STATUS_ACMP))



/* EPWM lock functions. */
void                    EPWM_Unlock(void);
void                    EPWM_Lock(void);

/* Deinitializes EPWM */
void                    EPWM_DeInit(void);

/* EPWM base time functions. */
void                    EPWM_TimeBaseInit(EPWM_TimeBaseInitTypeStruct* EPWM_TimeBaseInitType);
void                    EPWM_TimeBaseStructInit(EPWM_TimeBaseInitTypeStruct* EPWM_TimeBaseInitType);
void                    EPWM_Cmd(FunctionalState NewState);
void                    EPWM_SetAutoReload(uint16_t EPWM_Value);
uint16_t                EPWM_GetAutoReload(void);
uint16_t                EPWM_GetCounter(void);

/* EPWM ADC compare functions. */
void                    EPWM_ADCCmpInit(EPWM_ADCCmpInitTypeStruct* EPWM_ADCCmpInitType);
void                    EPWM_ADCCmpInitStruct(EPWM_ADCCmpInitTypeStruct* EPWM_ADCCmpInitType);
void                    EPWM_SetADCCompare1Value(uint16_t EPWM_Value);
void                    EPWM_SetADCCompare2Value(uint16_t EPWM_Value);
void                    EPWM_SetADCCompare1Dir(EPWM_CmpDirEnum EPWM_Dir);
void                    EPWM_SetADCCompare2Dir(EPWM_CmpDirEnum EPWM_Dir);
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
void                    EPWM_ADCCompare1Cmd(FunctionalState NewState);
void                    EPWM_ADCCompare2Cmd(FunctionalState NewState);
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

/* EPWM CH0,CH1.CH2.CH3 compare functions. */
void                    EPWM_CHInit(EPWM_ChannelEnum EPWM_Channel, EPWM_CHInitTypeStruct* EPWM_CHInitType);
void                    EPWM_CHStructInit(EPWM_CHInitTypeStruct* EPWM_CHInitType);
void                    EPWM_SetCHCompare1Value(EPWM_ChannelEnum EPWM_Channel, uint16_t EPWM_Value);
void                    EPWM_SetCHCompare2Value(EPWM_ChannelEnum EPWM_Channel, uint16_t EPWM_Value);
void                    EPWM_SetCHCompare1Action(EPWM_ChannelEnum EPWM_Channel, EPWM_CmpActEnum EPWM_Act);
void                    EPWM_SetCHCompare2Action(EPWM_ChannelEnum EPWM_Channel, EPWM_CmpActEnum EPWM_Act);
void                    EPWM_SetCHInitOutputLevel(EPWM_ChannelEnum EPWM_Channel, EPWM_CHInitStatusEnum EPWM_InitStatus);
void                    EPWM_SetCHNPolarity(EPWM_ChannelEnum EPWM_Channel, EPWM_CHOutPolarityEnum EPWM_Pol);
void                    EPWM_SetCHPPolarity(EPWM_ChannelEnum EPWM_Channel, EPWM_CHOutPolarityEnum EPWM_Pol);
void                    EPWM_SetCHCompare1Dir(EPWM_ChannelEnum EPWM_Channel, EPWM_CmpDirEnum EPWM_Dir);
void                    EPWM_SetCHCompare2Dir(EPWM_ChannelEnum EPWM_Channel, EPWM_CmpDirEnum EPWM_Dir);

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
void                    EPWM_CHCompare1Cmd(EPWM_ChannelEnum EPWM_Channel, FunctionalState NewState);
void                    EPWM_CHCompare2Cmd(EPWM_ChannelEnum EPWM_Channel, FunctionalState NewState);
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

void                    EPWM_CHCmd(EPWM_ChannelEnum EPWM_Channel, FunctionalState NewState);

/* EPWM CH0,CH1.CH2.CH3 force output functions. */
void                    EPWM_CHForceOutInit(EPWM_ChannelEnum EPWM_Channel, EPWM_FOInitTypeStruct* EPWM_FOtInitType);
void                    EPWM_CHForceOutStructInit(EPWM_FOInitTypeStruct* EPWM_FOtInitType);
void                    EPWM_SetCHNForceLevel(EPWM_ChannelEnum EPWM_Channel, EPWM_FOLevelEnum EPWM_Level);
void                    EPWM_SetCHPForceLevel(EPWM_ChannelEnum EPWM_Channel, EPWM_FOLevelEnum EPWM_Level);
void                    EPWM_CHNForceCmd(EPWM_ChannelEnum EPWM_Channel, FunctionalState NewState);
void                    EPWM_CHPForceCmd(EPWM_ChannelEnum EPWM_Channel, FunctionalState NewState);

/* EPWM update shadow register functions. */
void                    EPWM_UpdateShadowRegister(void);

/* EPWM config death time length functions. */
void                    EPWM_DeathTimeLenConfig(uint16_t EPWM_Value);

/* EPWM emergency stop functions. */
void                    EPWM_ESInit(EPWM_ESInitTypeStruct* EPWM_ESInitType);
void                    EPWM_ESStructInit(EPWM_ESInitTypeStruct* EPWM_ESInitType);
void                    EPWM_SetESFiltSampleDiv(EPWM_ESFiltSampleDivEnum EPWM_Div);
void                    EPWM_SetESFiltLen(EPWM_ESFiltLenEnum EPWM_Len);
void                    EPWM_SetESExtPolarity(EPWM_ESExtPolarityEnum EPWM_Pol);
void                    EPWM_ESExtCmd(FunctionalState NewState);
void                    EPWM_SetESACMPPolarity(EPWM_ESACMPPolarityEnum EPWM_Pol);
void                    EPWM_SetESACMPChannel(EPWM_ESACMPChannelEnum EPWM_Chl);
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
void                    EPWM_SetESACMPMode(EPWM_ESACMPModeEnum EPWM_ACMPMode);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

void                    EPWM_ESACMPCmd(FunctionalState NewState);
void                    EPWM_ESDebugCmd(FunctionalState NewState);
void                    EPWM_SetESCHNLevel(EPWM_ChannelEnum EPWM_Channel, EPWM_ESOutLevelEnum EPWM_Level);
void                    EPWM_SetESCHPLevel(EPWM_ChannelEnum EPWM_Channel, EPWM_ESOutLevelEnum EPWM_Level);
FlagState               EPWM_GetESStopStatus(EPWM_ESStopStatusEnum EPWM_ESStop);
void                    EPWM_ESSoftwareBrake(void);
void                    EPWM_ESSoftwareContinue(void);



#if defined (DPM32M08x) || defined (DPM32M05x)
/* EPWM DMA functions. */
void                    EPWM_DMACmd(FunctionalState NewState);
#endif  /* DPM32M08x || DPM32M05x */

/* EPWM interrupr functions. */
void                    EPWM_IntCmd(uint16_t EPWM_IntType, FunctionalState NewState);
FunctionalState         EPWM_GetIntCmdStatus(EPWM_IntTypeEnum EPWM_IntType);
FlagState               EPWM_GetIntFlagStatus(EPWM_IntFlagEnum EPWM_IntFlag);
void                    EPWM_ClearIntFlag(EPWM_IntFlagEnum EPWM_IntFlag);

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif


#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_EPWM_H__ */
