/**
 *******************************************************************************
 * @file    dpm32m0xx_pga.h
 *
 * @brief   Header file for PGA firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_PGA_H__
#define __DPM32M0XX_PGA_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif

#include "dpm32m0xx.h"

/**
 ******************************************************************************
 * @brief Programmable gain amplifier(PGA) channel enumeration.
 *****************************************************************************/
typedef enum
{
  PGA_CH0                = 0x00UL,    /*!< Programmable gain amplifier 0. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036)|| defined (DPM32M030)
  PGA_CH1                = 0x01UL,    /*!< Programmable gain amplifier 1. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M036)
  PGA_CH2                = 0x02UL,    /*!< Programmable gain amplifier 2. */
#endif /* DPM32M08x || DPM32M05x || DPM32M036 */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  PGA_CH3                = 0x03UL     /*!< Programmable gain amplifier 3. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} PGA_ChannelEnum;

/**
 ******************************************************************************
 * @brief Programmable gain amplifier(PGA) mode enumeration.
 *****************************************************************************/
typedef enum
{
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
  PGA_INPUT_MODE_SINGLE           = 0x00UL,         /*!< PGA single input mode. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
  PGA_INPUT_MODE_DIFFERENTIAL     = 0x01UL          /*!< PGA differential input mode. */
} PGA_InputModeEnum;


#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015) 
/**
 ******************************************************************************
 * @brief Programmable gain amplifier(PGA) amplify programmable gain enumeration.
 *****************************************************************************/
typedef enum
{
  PGA_AMP_GAIN1                    = 0x00UL,   /*!< PGA programmable gain is 1. */
  PGA_AMP_GAIN3,                               /*!< PGA programmable gain is 3. */
  PGA_AMP_GAIN5,                               /*!< PGA programmable gain is 5. */
  PGA_AMP_GAIN9,                               /*!< PGA programmable gain is 9. */
  PGA_AMP_GAIN17,                              /*!< PGA programmable gain is 17. */
  PGA_AMP_GAIN33                               /*!< PGA programmable gain is 33. */
} PGA_AmpGainEnum;
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
/**
 ******************************************************************************
 * @brief Programmable gain amplifier(PGA) amplify programmable gain enumeration.
 *****************************************************************************/
typedef enum
{
  PGA_AMP_GAIN1                    = 0x00UL,   /*!< PGA programmable gain is 1. */
  PGA_AMP_GAIN2,                               /*!< PGA programmable gain is 2. */
  PGA_AMP_GAIN4,                               /*!< PGA programmable gain is 4. */
  PGA_AMP_GAIN8,                               /*!< PGA programmable gain is 8. */
  PGA_AMP_GAIN16,                              /*!< PGA programmable gain is 16. */
  PGA_AMP_GAIN32                               /*!< PGA programmable gain is 32. */
} PGA_AmpGainEnum;

#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 ******************************************************************************
 * @brief Output bias select enumeration.
 *****************************************************************************/
typedef enum
{
  PGA_OFFSET_SEL_AVDD            = 0x00UL,     /*!< Select 1/2 AVDD as bias value. */
  PGA_OFFSET_SEL_BANDGAP                       /*!< Select 1/2 Bandgap as bias value. */
} PGA_OffsetSelEnum;

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
/**
 ******************************************************************************
 * @brief Output filter circuit input source selection enumeration.
 *****************************************************************************/
typedef enum
{
  PGA_FILT_SEL_PGA0            = 0x00UL,     /*!< Select PGA0 output as the input for the filter circuit. */
#if defined (DPM32M036) || defined (DPM32M030) 
  PGA_FILT_SEL_PGA1,                         /*!< Select PGA1 output as the input for the filter circuit. */
#endif /* DPM32M036 || DPM32M030 || */
#if defined (DPM32M036)
  PGA_FILT_SEL_PGA2                          /*!< Select PGA2 output as the input for the filter circuit. */
#endif /* DPM32M036 */
} PGA_FiltSelEnum;
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 ******************************************************************************
 * @brief Programmable gain amplifier configure structure.
 *****************************************************************************/
typedef struct
{
  PGA_InputModeEnum         PGA_Mode;                 /*!< PGA input mode configure.
                                                           This parameter can be a value of @ref PGA_InputModeEnum.*/
  PGA_AmpGainEnum           PGA_Gain;                 /*!< PGA amplify gain configure.
                                                           This parameter can be a value of @ref PGA_AmpGainEnum.*/
  PGA_OffsetSelEnum         PGA_OffsetSel;            /*!< PGA bias output value configure.
                                                           This parameter can be a value of @ref PGA_OffsetSelEnum.*/
  FunctionalState           PGA_OffsetState;          /*!< Specifies PGA bias output state.
                                                           This parameter can be a value of @ref FunctionalState.*/
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
  FunctionalState           PGA_FiltState;            /*!< Specifies PGA output filter enable.
                                                           This parameter can be a value of @ref FunctionalState.*/
  PGA_FiltSelEnum           PGA_FiltSel;              /*!< PGA output filter circuit input source selection.
                                                           This parameter can be a value of @ref PGA_FiltSelEnum.*/
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

} PGA_InitTypeStruct;


/* PGA param check. */
#if defined (DPM32M015)
#define IS_PGA_CHANNEL(x) (((x) == PGA_CH0))
#endif /* DPM32M015 */

#if defined (DPM32M030)
#define IS_PGA_CHANNEL(x) (((x) == PGA_CH0) || \
                           ((x) == PGA_CH1))
#endif /* DPM32M030 */

#if defined (DPM32M03x)
#define IS_PGA_CHANNEL(x) (((x) == PGA_CH0) || \
                           ((x) == PGA_CH1) || \
                           ((x) == PGA_CH3))
#endif /* DPM32M03x */

#if defined (DPM32M036) 
#define IS_PGA_CHANNEL(x) (((x) == PGA_CH0) || \
                           ((x) == PGA_CH1) || \
                           ((x) == PGA_CH2)) 
#endif /* DPM32M036 */

#if defined (DPM32M08x) || defined (DPM32M05x) 
#define IS_PGA_CHANNEL(x) (((x) == PGA_CH0) || \
                           ((x) == PGA_CH1) || \
                           ((x) == PGA_CH2) || \
                           ((x) == PGA_CH3))
#endif /* DPM32M08x || DPM32M05x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_PGA_INOUT_MODE(x) (((x) == PGA_INPUT_MODE_SINGLE)|| \
                              ((x) == PGA_INPUT_MODE_DIFFERENTIAL))
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_PGA_INOUT_MODE(x) (((x) == PGA_INPUT_MODE_DIFFERENTIAL))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
#define IS_PGA_AMP_GAIN(x) (((x) == PGA_AMP_GAIN1) || \
                            ((x) == PGA_AMP_GAIN3) || \
                            ((x) == PGA_AMP_GAIN5) || \
                            ((x) == PGA_AMP_GAIN9) || \
                            ((x) == PGA_AMP_GAIN17) || \
                            ((x) == PGA_AMP_GAIN33))
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_PGA_AMP_GAIN(x) (((x) == PGA_AMP_GAIN1) || \
                            ((x) == PGA_AMP_GAIN2) || \
                            ((x) == PGA_AMP_GAIN4) || \
                            ((x) == PGA_AMP_GAIN8) || \
                            ((x) == PGA_AMP_GAIN16) || \
                            ((x) == PGA_AMP_GAIN32))
#endif /* DPM32M036 */

#define IS_PGA_OFFSET_SEL(x) (((x) == PGA_OFFSET_SEL_AVDD) || \
                              ((x) == PGA_OFFSET_SEL_BANDGAP))

#if defined (DPM32M036)
#define IS_PGA_FILT_SEL(x) (((x) == PGA_FILT_SEL_PGA0) || \
                            ((x) == PGA_FILT_SEL_PGA1) || \
                            ((x) == PGA_FILT_SEL_PGA2))
#endif /* DPM32M036 */

#if defined (DPM32M030) 
#define IS_PGA_FILT_SEL(x) (((x) == PGA_FILT_SEL_PGA0) || \
                            ((x) == PGA_FILT_SEL_PGA1))
#endif /* DPM32M030 */

#if defined (DPM32M015)
#define IS_PGA_FILT_SEL(x) (((x) == PGA_FILT_SEL_PGA0))
#endif /* DPM32M015 */


/* Deinitializes,initialization and cmd functions. */
void                    PGA_DeInit(PGA_ChannelEnum PGA_Channel);
void                    PGA_Init(PGA_ChannelEnum PGA_Channel, PGA_InitTypeStruct* PGA_InitType);
void                    PGA_StructInit(PGA_InitTypeStruct* PGA_InitType);
void                    PGA_Cmd(PGA_ChannelEnum PGA_Channel, FunctionalState NewState);

/* PGA Output bias enable and select functions. */
void                    PGA_OffsetCmd(PGA_ChannelEnum PGA_Channel, FunctionalState NewState);
void                    PGA_OffsetSelect(PGA_OffsetSelEnum PGA_OffsetSel);

/* PGA channel configuration functions. */
void                    PGA_SetInputMode(PGA_ChannelEnum PGA_Channel, PGA_InputModeEnum PGA_Mode);
void                    PGA_SetAmpGain(PGA_ChannelEnum PGA_Channel, PGA_AmpGainEnum PGA_Gain);

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    PGA_FiltCmd(FunctionalState NewState);
void                    PGA_FiltSelect(PGA_FiltSelEnum PGA_Filt);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif


#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_PGA_H__ */
