/**
 *******************************************************************************
 * @file    dpm32m0xx_dev.c
 *
 * @brief   Source file for DES firmware driver.
 *          This file provides firmware functions to manage the following :
 *          functionalities of the Device Electronic Signature (DES) peripheral:
 *           + Information management
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 * @verbatim
 ===============================================================================
                       ##### How to use this driver #####
 ===============================================================================
    [..]
      (#) Get unique device ID word using DES_GetUIDWord0(), DES_GetUIDWord1()
          and DES_GetUIDWord2().

      (#) Get device information using DES_GetDevInfo().

 * @endverbatim
 *******************************************************************************/


#include "dpm32m0xx_des.h"



/**
 *******************************************************************************
 * @brief   Return first word of the unique device identifier (UID based on 96 bits).
 * @retval  uint32_t: Unique device identifer.
 ******************************************************************************/
uint32_t DES_GetUIDWord0(void)
{
  return DES->UID_WORD0;
}

/**
 *******************************************************************************
 * @brief   Return second word of the unique device identifier (UID based on 96 bits).
 * @retval  uint32_t: Unique device identifer.
 ******************************************************************************/
uint32_t DES_GetUIDWord1(void)
{
  return DES->UID_WORD1;
}

/**
 *******************************************************************************
 * @brief   Return third word of the unique device identifier (UID based on 96 bits).
 * @retval  uint32_t: Unique device identifer.
 ******************************************************************************/
uint32_t DES_GetUIDWord2(void)
{
  return DES->UID_WORD2;
}

/**
 *******************************************************************************
 * @brief   Return the RAM size.
 * @retval  DES_RAMSizeEnum: RAM size.
 *          This parameter can be a value of @ref DES_RAMSizeEnum.
 *            @arg DES_RAM_SIZE_1KB: SRAM is 1KB.
 *            @arg DES_RAM_SIZE_2KB: SRAM is 2KB.
 *            @arg DES_RAM_SIZE_4KB: SRAM is 4KB.
 *            @arg DES_RAM_SIZE_8KB: SRAM is 8KB.
 *            @arg DES_RAM_SIZE_16KB: SRAM is 16KB.
 *            @arg DES_RAM_SIZE_32KB: SRAM is 32KB.
 *            @arg DES_RAM_SIZE_64KB: SRAM is 64KB.
 *            @arg DES_RAM_SIZE_128KB: SRAM is 128KB.
 ******************************************************************************/
DES_RAMSizeEnum DES_GetRAMSize(void)
{
  return (DES_RAMSizeEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_RAM_SIZE_Msk) \
                                 >> DES_DEV_INFO1_RAM_SIZE_Pos);
}

/**
 *******************************************************************************
 * @brief   Return Product type.
 * @retval  DES_ProdTypeEnum: prod type.
 *          This parameter can be a value of @ref DES_ProdTypeEnum.
 *            @arg DES_PROD_TYPE_F: F series, general purpose.
 *            @arg DES_PROD_TYPE_L: L series, low power consumption.
 *            @arg DES_PROD_TYPE_H: H series, high performance.
 *            @arg DES_PROD_TYPE_M: M series, motor driven.
 ******************************************************************************/
DES_ProdTypeEnum DES_GetProdType(void)
{
  return (DES_ProdTypeEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_PROD_TYPE_Msk) \
                                  >> DES_DEV_INFO0_PROD_TYPE_Pos);
}

/**
 *******************************************************************************
 * @brief   Return device sub family coding.
 * @retval  uint16_t: coding.
 ******************************************************************************/
uint16_t DES_GetSubFamily(void)
{
  return (uint16_t)((DES->DEV_INFO0 & DES_DEV_INFO0_SUB_FAM_Msk) >> DES_DEV_INFO0_SUB_FAM_Pos);
}

/**
 *******************************************************************************
 * @brief   Return pin numer.
 * @retval  DES_PinNumEnum: Pin number.
 *          This parameter can be a value of @ref DES_PinNumEnum.
 *            @arg DES_PIN_NUM_8: pins is 8.
 *            @arg DES_PIN_NUM_16: pins is 16.
 *            @arg DES_PIN_NUM_20: pins is 20.
 *            @arg DES_PIN_NUM_24: pins is 24.
 *            @arg DES_PIN_NUM_28: pins is 28.
 *            @arg DES_PIN_NUM_32: pins is 32.
 *            @arg DES_PIN_NUM_40: pins is 40.
 *            @arg DES_PIN_NUM_48: pins is 48.
 *            @arg DES_PIN_NUM_52: pins is 52.
 *            @arg DES_PIN_NUM_64: pins is 64.
 *            @arg DES_PIN_NUM_100: pins is 100.
 *            @arg DES_PIN_NUM_144: pins is 144.
 ******************************************************************************/
DES_PinNumEnum DES_GetPinNumer(void)
{
  return (DES_PinNumEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_PIN_NUM_Msk) \
                                  >> DES_DEV_INFO0_PIN_NUM_Pos);
}

/**
 *******************************************************************************
 * @brief   Return the flash size.
 * @retval  DES_FlashSizeEnum: Flash size.
 *          This parameter can be a value of @ref DES_SRAMSizeEnum.
 *            @arg DES_FLASH_SIZE_4KB: Flash is 4KB.
 *            @arg DES_FLASH_SIZE_8KB: Flash is 8KB.
 *            @arg DES_FLASH_SIZE_16KB: Flash is 16KB.
 *            @arg DES_FLASH_SIZE_32KB: Flash is 32KB.
 *            @arg DES_FLASH_SIZE_64KB: Flash is 64KB.
 *            @arg DES_FLASH_SIZE_128KB: Flash is 128KB.
 *            @arg DES_FLASH_SIZE_256KB: Flash is 256KB.
 *            @arg DES_FLASH_SIZE_512KB: Flash is 512KB.
 ******************************************************************************/
DES_FlashSizeEnum DES_GetFlashSize(void)
{
  return (DES_FlashSizeEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_FLASH_SIZE_Msk) \
                                   >> DES_DEV_INFO0_FLASH_SIZE_Pos);
}

/**
 *******************************************************************************
 * @brief   Return the package type.
 * @retval  DES_PackTypeEnum: Package type.
 *          This parameter can be a value of @ref DES_PackTypeEnum.
 *            @arg DES_PACK_TYPE_TSSOP.
 *            @arg DES_PACK_TYPE_SOP.
 *            @arg DES_PACK_TYPE_LQPF.
 *            @arg DES_PACK_TYPE_QPN.
 ******************************************************************************/
DES_PackTypeEnum DES_GetPackType(void)
{
  return (DES_PackTypeEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_PKG_TYPE_Msk) \
                                  >> DES_DEV_INFO0_PKG_TYPE_Pos);
}

/**
 *******************************************************************************
 * @brief   Return integrated Driver Type.
 * @retval  DES_GateDriverEnum: Integrated Driver Type.
 *          This parameter can be a value of @ref DES_GateDriverEnum.
 *            @arg DES_GATE_DRIVER_NONE.
 *            @arg DES_GATE_DRIVER_6N.
 *            @arg DES_GATE_DRIVER_3P3N.
 *            @arg DES_GATE_DRIVER_3P3N_MOS.
 *            @arg DES_GATE_DRIVER_6N_MOS.
 ******************************************************************************/
DES_GateDriverEnum DES_GetGateDriver(void)
{
  return (DES_GateDriverEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_GATE_DRIVER_Msk) \
                                  >> DES_DEV_INFO1_GATE_DRIVER_Pos);
}

/**
 *******************************************************************************
 * @brief   Return integrated Power Type .
 * @retval  DES_IntLDOEnum: Integrated Power Type.
 *          This parameter can be a value of @ref DES_IntLDOEnum.
 *            @arg DES_INT_LDO_NONE.
 *            @arg DES_INT_LDO_5.
 *            @arg DES_INT_LDO_3V3.
 *            @arg DES_INT_LDO_12_5.
 *            @arg DES_INT_LDO_12_3V3.
 *            @arg DES_INT_LDO_15_5.
 *            @arg DES_INT_LDO_15_3V3.

 ******************************************************************************/
DES_IntLDOEnum DES_GetIntLDO(void)
{
  return (DES_IntLDOEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_INT_LDO_Msk) \
                                  >> DES_DEV_INFO1_INT_LDO_Pos);
}

/**
 *******************************************************************************
 * @brief   Return integrated Driver Type.
 * @retval  DES_TemperatureEnum: Integrated Driver Type.
 *          This parameter can be a value of @ref DES_TemperatureEnum.
 *            @arg DES_TEMP_N40_P125.
 *            @arg DES_TEMP_N40_P85.
 *            @arg DES_TEMP_N40_P105.
 ******************************************************************************/
DES_TemperatureEnum DES_GetTemperature(void)
{
  return (DES_TemperatureEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_TEMP_Msk) \
                                  >> DES_DEV_INFO1_TEMP_Pos);
}

/**
 *******************************************************************************
 * @brief   Return Integrated Driver/Power Version Information.
 * @retval  uint8_t: Integrated Driver/Power Version Information.
 ******************************************************************************/
uint8_t DES_GetIntVersion(void)
{
  return (uint8_t)((DES->DEV_INFO1 & DES_DEV_INFO1_INT_VERSION_Msk) >> DES_DEV_INFO1_INT_VERSION_Pos);
}

/**
 *******************************************************************************
 * @brief   Return Package Pinout Version Information.
 * @retval  uint8_t: Package Pinout Version Information.
 ******************************************************************************/
uint8_t DES_GetPackVersion(void)
{
  return (uint8_t)((DES->DEV_INFO1 & DES_DEV_INFO1_PKG_VERSION_Msk) >> DES_DEV_INFO1_PKG_VERSION_Pos);
}

/**
 *******************************************************************************
 * @brief   Return MCU Version Information.
 * @retval  uint8_t: MCU Version Information.
 ******************************************************************************/
uint8_t DES_GetMCUVersion(void)
{
  return (uint8_t)((DES->DEV_INFO1 & DES_DEV_INFO1_MCU_VERSION_Msk) >> DES_DEV_INFO1_MCU_VERSION_Pos);
}

/**
 *******************************************************************************
 * @brief   Get device informations.
 * @param   [in]  DES_DevInfoType Structure pointer of device informations.
 * @retval  None.
 ******************************************************************************/
void DES_GetDevInfo(DES_DevInfoTypeStruct* DES_DevInfoType)
{
  DES_DevInfoType->DES_UID[0] = DES->UID_WORD0;
  DES_DevInfoType->DES_UID[1] = DES->UID_WORD1;
  DES_DevInfoType->DES_UID[2] = DES->UID_WORD2;
  DES_DevInfoType->DES_ProdType = (DES_ProdTypeEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_PROD_TYPE_Msk) \
                                  >> DES_DEV_INFO0_PROD_TYPE_Pos);
  DES_DevInfoType->DES_SubFamily = (uint16_t)((DES->DEV_INFO0 & DES_DEV_INFO0_SUB_FAM_Msk) \
                                   >> DES_DEV_INFO0_SUB_FAM_Pos);
  DES_DevInfoType->DES_PinNumer = (DES_PinNumEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_PIN_NUM_Msk) \
                                  >> DES_DEV_INFO0_PIN_NUM_Pos);
  DES_DevInfoType->DES_FlashSize = (DES_FlashSizeEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_FLASH_SIZE_Msk) \
                                   >> DES_DEV_INFO0_FLASH_SIZE_Pos);
  DES_DevInfoType->DES_PackType = (DES_PackTypeEnum)((DES->DEV_INFO0 & DES_DEV_INFO0_PKG_TYPE_Msk) \
                                  >> DES_DEV_INFO0_PKG_TYPE_Pos);

  DES_DevInfoType->DES_GateDriver = (DES_GateDriverEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_GATE_DRIVER_Msk) \
                                  >> DES_DEV_INFO1_GATE_DRIVER_Pos);
  DES_DevInfoType->DES_IntLDO = (DES_IntLDOEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_INT_LDO_Msk) \
                                  >> DES_DEV_INFO1_INT_LDO_Pos);
  DES_DevInfoType->DES_IntVersion = (uint8_t)((DES->DEV_INFO1 & DES_DEV_INFO1_INT_VERSION_Msk) \
                                 >> DES_DEV_INFO1_INT_VERSION_Pos);
  DES_DevInfoType->DES_Temperature = (DES_TemperatureEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_TEMP_Msk) \
                                  >> DES_DEV_INFO1_TEMP_Pos);
  DES_DevInfoType->DES_PackVersion = (uint8_t)((DES->DEV_INFO1 & DES_DEV_INFO1_PKG_VERSION_Msk) \
                                 >> DES_DEV_INFO1_PKG_VERSION_Pos);
  DES_DevInfoType->DES_MCUVersion = (uint8_t)((DES->DEV_INFO1 & DES_DEV_INFO1_MCU_VERSION_Msk) \
                                 >> DES_DEV_INFO1_MCU_VERSION_Pos);
  DES_DevInfoType->DES_RAMSize = (DES_RAMSizeEnum)((DES->DEV_INFO1 & DES_DEV_INFO1_RAM_SIZE_Msk) \
                                 >> DES_DEV_INFO1_RAM_SIZE_Pos);
}

