/**
 *******************************************************************************
 * @file    dpm32m0xx_wdg.h
 *
 * @brief   Header file for WDG firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_WDG_H__
#define __DPM32M0XX_WDG_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif

#include "dpm32m0xx.h"


#define WDGKEY           0X900D0000UL
#define FEEDDOG          0x900DBEEFUL

/**
 ******************************************************************************
 * @brief WDG mode enumeration.
 *****************************************************************************/
typedef enum
{
  WDG_MODE_7BIT         = 0x00UL,                   /*!< Maximum count value 128 selection. 3.90625ms reset. */
  WDG_MODE_8BIT         = 0x10UL,                   /*!< Maximum count value 256 selection. 7.8125ms reset. */
  WDG_MODE_9BIT         = 0x20UL,                   /*!< Maximum count value 512 selection. 15.625ms reset. */
  WDG_MODE_10BIT        = 0x30UL,                   /*!< Maximum count value 1024 selection. 31.25ms reset. */
  WDG_MODE_11BIT        = 0x40UL,                   /*!< Maximum count value 2048 selection. 62.5ms reset. */
  WDG_MODE_12BIT        = 0x50UL,                   /*!< Maximum count value 4096 selection. 125ms reset. */
  WDG_MODE_13BIT        = 0x60UL,                   /*!< Maximum count value 8192 selection. 250ms reset. */
  WDG_MODE_14BIT        = 0x70UL,                   /*!< Maximum count value 16384 selection. 500ms reset. */
  WDG_MODE_15BIT        = 0x80UL,                   /*!< Maximum count value 32768 selection. 1s reset. */
  WDG_MODE_16BIT        = 0x90UL,                   /*!< Maximum count value 65536 selection. 2s reset. */
  WDG_MODE_17BIT        = 0xA0UL,                   /*!< Maximum count value 131072 selection. 4s reset. */
  WDG_MODE_18BIT        = 0xB0UL,                   /*!< Maximum count value 262144 selection. 8s reset. */
  WDG_MODE_19BIT        = 0xC0UL,                   /*!< Maximum count value 524288 selection. 16s reset. */
  WDG_MODE_20BIT        = 0xD0UL,                   /*!< Maximum count value 1048576 selection. 32s reset. */
  WDG_MODE_21BIT        = 0xE0UL,                   /*!< Maximum count value 2097152 selection. 64s reset. */
  WDG_MODE_22BIT        = WDG_CSR_MODE_Msk          /*!< Maximum count value 4194304 selection. 128s reset. */
} WDG_ModeEnum;

/* WDG MODE Check. */
#define IS_WDG_MODE(x) (((x) == WDG_MODE_7BIT) || \
                        ((x) == WDG_MODE_8BIT) || \
                        ((x) == WDG_MODE_9BIT) || \
                        ((x) == WDG_MODE_10BIT) || \
                        ((x) == WDG_MODE_11BIT) || \
                        ((x) == WDG_MODE_12BIT) || \
                        ((x) == WDG_MODE_13BIT) || \
                        ((x) == WDG_MODE_14BIT) || \
                        ((x) == WDG_MODE_15BIT) || \
                        ((x) == WDG_MODE_16BIT) || \
                        ((x) == WDG_MODE_17BIT) || \
                        ((x) == WDG_MODE_18BIT) || \
                        ((x) == WDG_MODE_19BIT) || \
                        ((x) == WDG_MODE_20BIT) || \
                        ((x) == WDG_MODE_21BIT) || \
                        ((x) == WDG_MODE_22BIT))

/**
 ******************************************************************************
 * @brief WDG Init Struct.
 *****************************************************************************/
typedef struct
{
  WDG_ModeEnum        WDG_Mode;             /*!< Time interval for counter overflow. */
} WDG_InitTypeStruct;


/* Deinitializes and initialization functions. */
void                    WDG_DeInit(void);
void                    WDG_Init(WDG_InitTypeStruct *WDG_InitType);
void                    WDG_Cmd(FunctionalState NewState);
void                    WDG_Enable(void);
void                    WDG_Disable(void);

/* WDG counter mode configuration and feeddog functions. */
void                    WDG_CounterModeConfig(WDG_ModeEnum WDG_Mode);
void                    WDG_FeedDog(void);

/* WDG interrupt related functions. */
void                    WDG_IntCmd(FunctionalState NewState);
FunctionalState         WDG_GetIntCmdStatus(void);
FlagState               WDG_GetIntFlagStatus(void);
void                    WDG_ClearIntFlag(void);

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
void                    WDG_MaskEnable(void);
FlagState               WDG_GetMaskStatus(void);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_WDG_H__ */
