/**
 *******************************************************************************
 * @file    dpm32m0xx_spi.c
 *
 * @brief   Source file for SPI firmware driver.
 *          This file provides firmware functions to manage the following
 *          functionalities of the Serial peripheral interface (SPI):
 *           + Initialization and Configuration
 *           + Data transfers functions
 *           + DMA transfers management
 *           + Interrupts and flags management
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 * @verbatim
 ===============================================================================
                       ##### How to use this driver #####
 ===============================================================================
    [..]
      (#) Enable peripheral clock using the following functions
          RCC_APBPeriphClockCmd(RCC_APB_PERIPH_SPIx, ENABLE).

      (#) Enable SCK, MOSI, MISO and NSS GPIO clocks using
          RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE) function.

      (#) Peripheral's alternate function:
        (+) Connect the pin to the desired peripherals' Alternate
            Function (AF) using GPIO_AltFuncConfig() function.
        (+) Configure the desired pin in alternate function by:
            GPIO_InitStruct->GPIO_Mode = GPIO_MODE_ALT_FUNC.
        (+) Select the type, pull-up/pull-down and output speed via
            GPIO_PuPd, GPIO_OType and GPIO_Speed members.
        (+) Call GPIO_Init() function.

      (#) Program the Peripheral Mode, Data size, Polarity, Baud Rate Prescaler,
          First Bit, Slave Management using the SPI_Init() function.

      (#) Enable the NVIC and the corresponding interrupt using the function
          SPI_IntCmd() if you need to use interrupt mode.

      (#) When using the DMA mode.
        (+) Configure the DMA using DMA_Init() function.
        (+) Active the needed channel Request using SPI_DMACmd() function.

      (#) Enable the SPI using the SPI_Cmd() function.

      (#) Enable the DMA using the DMA_Cmd() function when using DMA mode.

      (#) You can management Transmit and Receive using SPI_TransmitCmd() and
          SPI_ReceiveCmd() function;

      (#) Data transfers using the SPI_SendData() and SPI_ReceiveData() function.

      (#) SPI interrupt configuration:
        (+) To activate the SPI interrupt, using SPI_IntCmd() functions.
        (+) Check on SPI interrupt enable flags using the function
            SPI_GetIntCmdStatus().
        (+) Check on SPI interrupt occur flags using the function
            SPI_GetIntFlagStatus().
        (+) Clear SPI interrupt flags using the function SPI_ClearIntFlag().

 * @endverbatim
 *******************************************************************************/

#include "dpm32m0xx_spi.h"



/**
 *******************************************************************************
 * @brief  Deinitialize the SPIx peripheral registers to their default reset values.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @retval None.
 ******************************************************************************/
void SPI_DeInit(SPI_Type *SPIx)
{
  /* check the parameters */
  PARAM_ASSERT(IS_SPI_PERIPH(SPIx));

  /* Reset SPIx CR register. */
  SPIx->CR = (uint32_t)0x00000700;
  /* Clear all interrupt flag */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  SPIx->ISR = ((uint32_t)(SPI_ISR_UDRR_Msk | SPI_ISR_FAULT_ERR_Msk | SPI_ISR_OVRR_ERR_Msk \
                          | SPI_ISR_TXE_Msk | SPI_ISR_RXNE_Msk));
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
  
#if defined (DPM32M036)
  SPIx->ISR = ((uint32_t)(SPI_ISR_FAULT_ERR_Msk | SPI_ISR_OVRR_ERR_Msk \
                          | SPI_ISR_TXE_Msk | SPI_ISR_RXNE_Msk));
#endif /* DPM32M036 */

  /* Reset SPIx BRR register. */
  SPIx->BRR = (uint32_t)0x00000000;
  /* Reset SPIx TDR register. */
  SPIx->TDR = (uint32_t)0x00000000;
}

/**
 *******************************************************************************
 * @brief  Initializes the SPIx peripheral according to the specified.
 *         parameters in the SPI_InitType.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_InitType: Pointer to a SPI_InitTypeStruct structure that contains the
 *               configuration information for the specified SPI peripheral.
 * @retval None.
 ******************************************************************************/
void SPI_Init(SPI_Type *SPIx, SPI_InitTypeStruct *SPI_InitType)
{
  uint32_t tmpReg = 0UL;

  /* Check the parameters */
  PARAM_ASSERT(IS_SPI_PERIPH(SPIx));
  PARAM_ASSERT(IS_SPI_BAUDRATE_PRESCALER(SPI_InitType->SPI_BaudRatePrescaler));
  PARAM_ASSERT(IS_SPI_CPHA(SPI_InitType->SPI_CPHA));
  PARAM_ASSERT(IS_SPI_CPOL(SPI_InitType->SPI_CPOL));
  PARAM_ASSERT(IS_SPI_DATASIZE(SPI_InitType->SPI_DataSize));
  PARAM_ASSERT(IS_SPI_FIRST_BIT(SPI_InitType->SPI_FirstBit));
  PARAM_ASSERT(IS_SPI_MODE(SPI_InitType->SPI_Mode));
  PARAM_ASSERT(IS_SPI_NSS_SELECT(SPI_InitType->SPI_NSSSlectEn));

  /*---------------------------- SPIx CR Configuration ------------------------*/
  /* Get the SPIx CR value */
  tmpReg = SPIx->CR;

  /* Clear CPHA,CPOL,DATA_SIZE[3:0],LSB_FIRST,MASTER,NSS_SEL,TX_EN,RX_EN bits. */
  tmpReg &= ~(SPI_CR_CPHA_Msk | SPI_CR_CPOL_Msk | SPI_CR_DATA_SIZE_Msk | SPI_CR_LSB_FIRST_Msk \
              | SPI_CR_MASTER_Msk | SPI_CR_NSS_SEL_Msk | SPI_CR_TX_EN_Msk | SPI_CR_RX_EN_Msk);
  /* Set CPHA bits according to SPI_CPHA. */
  tmpReg |= (((uint32_t)SPI_InitType->SPI_CPHA) << SPI_CR_CPHA_Pos);

  /* Set CPOL bits according to SPI_CPOL. */
  tmpReg |= (((uint32_t)SPI_InitType->SPI_CPOL) << SPI_CR_CPOL_Pos);

  /* Set DATA_SIZE[3:0] bits according to SPI_DataSize. */
  tmpReg |= (((uint32_t)SPI_InitType->SPI_DataSize) << SPI_CR_DATA_SIZE_Pos);

  /* Set LSB_FIRST bits according to SPI_FirstBit. */
  tmpReg |= (((uint32_t)SPI_InitType->SPI_FirstBit) << SPI_CR_LSB_FIRST_Pos);

  /* Set MASTER bits according to SPI_Mode. */
  tmpReg |= (((uint32_t)SPI_InitType->SPI_Mode) << SPI_CR_MASTER_Pos);

  /* Set NSS_SEL bits according to SPI_NSSSlectEn. */
  tmpReg |= (((uint32_t)SPI_InitType->SPI_NSSSlectEn) << SPI_CR_NSS_SEL_Pos);

  /* Enable sending and receiving */
  tmpReg |=  SPI_CR_TX_EN_Msk;
  tmpReg |=  SPI_CR_RX_EN_Msk;

  /* Write to SPIx CR */
  SPIx->CR = tmpReg;

  /*---------------------------- SPIx BRR Configuration ------------------------*/
  /* Clear BRR_RATE[7:0] bits. */
  SPIx->BRR &= ~SPI_BRR_BRR_Msk;
  /* Set BRR[7:0] bits according to SPI_BaudRatePrescaler. */
  SPIx->BRR |= (((uint32_t)SPI_InitType->SPI_BaudRatePrescaler) << SPI_BRR_BRR_Pos);
}

/**
 *******************************************************************************
 * @brief  Initializes the SPIx peripheral according to the specified
 *         parameters in the SPI_InitType.
 * @param  [out] SPI_InitType: Pointer to a SPI_InitTypeStruct structure
 *               which will be initialized.
 * @retval None.
 ******************************************************************************/
void SPI_StructInit(SPI_InitTypeStruct *SPI_InitType)
{
  /*--------------- Reset SPI init structure parameters values -----------------*/
  /* initialize the SPI_Mode member */
  SPI_InitType->SPI_Mode = SPI_MODE_SLAVE;
  /* initialize the SPI_DataSize member */
  SPI_InitType->SPI_DataSize = SPI_DATASIZE_8BIT;
  /* Initialize the SPI_CPOL member */
  SPI_InitType->SPI_CPOL = SPI_CPOL_LOW;
  /* Initialize the SPI_CPHA member */
  SPI_InitType->SPI_CPHA = SPI_CPHA_1EDGE;
  /* Initialize the SPI_NSSSlectEn member */
  SPI_InitType->SPI_NSSSlectEn = SPI_NSS_SELECT_DISABLE;
  /* Initialize the SPI_BaudRatePrescaler member */
  SPI_InitType->SPI_BaudRatePrescaler = SPI_BAUDRATE_PRESCALER_2;
  /* Initialize the SPI_FirstBit member */
  SPI_InitType->SPI_FirstBit = SPI_FRIST_BIT_MSB;
}

/**
 *******************************************************************************
 * @brief  Enables or disables the specified SPI peripheral.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  Newstate: Newstate of the SPI peripheral.
 *               Any value of @ref FunctionalState.
 *                 @arg DISABLE: disbale SPI peripheral.
 *                 @arg ENABLE: enable SPI peripheral.
 * @retval None.
 ******************************************************************************/
void SPI_Cmd(SPI_Type *SPIx, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /* Enable the selected SPI peripheral. */
    SPIx->CR |= SPI_CR_EN_Msk;
  }
  else
  {
    /* Disable the selected SPI peripheral. */
    SPIx->CR &= ~SPI_CR_EN_Msk;
  }
}

/**
 *******************************************************************************
 * @brief  Configures the data size for the selected SPI.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_DataSize: specifies the SPI data size.
 *               Any value of @ref SPI_DataSizeEnum.
 *                 @arg SPI_DATASIZE_4BIT: Set data frame format to 4bit.
 *                 @arg SPI_DATASIZE_5BIT: Set data frame format to 5bit.
 *                 @arg SPI_DATASIZE_6BIT: Set data frame format to 6bit.
 *                 @arg SPI_DATASIZE_7BIT: Set data frame format to 7bit.
 *                 @arg SPI_DATASIZE_8BIT: Set data frame format to 8bit.
 *                 @arg SPI_DATASIZE_9BIT: Set data frame format to 9bit.
 *                 @arg SPI_DATASIZE_10BIT: Set data frame format to 10bit.
 *                 @arg SPI_DATASIZE_11BIT: Set data frame format to 11bit.
 *                 @arg SPI_DATASIZE_12BIT: Set data frame format to 12bit.
 *                 @arg SPI_DATASIZE_13BIT: Set data frame format to 13bit.
 *                 @arg SPI_DATASIZE_14BIT: Set data frame format to 14bit.
 *                 @arg SPI_DATASIZE_15BIT: Set data frame format to 15bit.
 *                 @arg SPI_DATASIZE_16BIT: Set data frame format to 16bit.
 * @retval None.
 ******************************************************************************/
void SPI_DataSizeConfig(SPI_Type *SPIx, SPI_DataSizeEnum SPI_DataSize)
{
  /* Clear DATA_SIZE[3:0] bits. */
  SPIx->CR &= ~SPI_CR_DATA_SIZE_Msk;
  /* Set new DATA_SIZE[3:0] bits value */
  SPIx->CR |= (((uint32_t)SPI_DataSize) << SPI_CR_DATA_SIZE_Pos);
}

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
/**
 *******************************************************************************
 * @brief   Configures the frame interval in master for the selected SPI.
 * @param   [in]  SPIx: Where x can be 0 to select the SPI peripheral.
 * @param   [in]  SPI_FrameInterval: Frame interval.
 *                This parameter can be a number between 0x00 and 0x0F.
 * @retval  None.
 ******************************************************************************/
void SPI_MasterFrameIntervalConfig(SPI_Type *SPIx, uint8_t SPI_FrameInterval)
{
  /* Clear FRAME_INR[3:0] bits. */
  SPIx->CR &= ~SPI_CR_FRAME_INR_Msk;
  /* Set new FRAME_INR[3:0] bits value */
  SPIx->CR |= (((uint32_t)SPI_FrameInterval) << SPI_CR_FRAME_INR_Pos);
}

/**
 *******************************************************************************
 * @brief   Enables or disables the specified SPI continuous receive master functions.
 * @param   [in]  SPIx: Where x can be 0 to select the SPI peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void SPI_MasterContinueRecvCmd(SPI_Type *SPIx, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /* Enable the selected SPI continuous receive master functions. */
    SPIx->CR |= SPI_CR_CONT_RCV_Msk;
  }
  else
  {
    /* Disable the selected SPI continuous receive master functions. */
    SPIx->CR &= ~SPI_CR_CONT_RCV_Msk;
  }
}

/**
 *******************************************************************************
 * @brief   Enables or disables the specified SPI high speed slave functions.
 * @param   [in]  SPIx: Where x can be 0 to select the SPI peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void SPI_SlaveHighSpeedCmd(SPI_Type *SPIx, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /* Enable the selected SPI high speed slave functions. */
    SPIx->CR |= SPI_CR_HS_SLAVE_Msk;
  }
  else
  {
    /* Disable the selected SPIhigh speed slave functions. */
    SPIx->CR &= ~SPI_CR_HS_SLAVE_Msk;
  }
}
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 *******************************************************************************
 * @brief  Enables or disables the specified SPI transmit functions.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  Newstate: New state of the SPI transmit functions.
 *               Any value of @ref FunctionalState.
 *                 @arg DISABLE: disbale SPI transmit functions.
 *                 @arg ENABLE: enable SPI transmit functions.
 * @retval None.
 ******************************************************************************/
void SPI_TransmitCmd(SPI_Type *SPIx, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /* Enable the selected SPI transmit functions. */
    SPIx->CR |= SPI_CR_TX_EN_Msk;
  }
  else
  {
    /* Disable the selected SPI transmit functions. */
    SPIx->CR &= ~SPI_CR_TX_EN_Msk;
  }
}

/**
 *******************************************************************************
 * @brief  Enables or disables the specified SPI receive functions.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  Newstate: New Newstate of the SPI receive functions.
 *               Any value of @ref FunctionalState.
 *                 @arg DISABLE: disbale SPI receive functions.
 *                 @arg ENABLE: enable SPI receive functions.
 * @retval None.
 ******************************************************************************/
void SPI_ReceiveCmd(SPI_Type *SPIx, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /* Enable the selected SPI receive functions. */
    SPIx->CR |= SPI_CR_RX_EN_Msk;
  }
  else
  {
    /* Disable the selected SPI receive functions. */
    SPIx->CR &= ~SPI_CR_RX_EN_Msk;
  }
}

/**
 *******************************************************************************
 * @brief  Transmits a Data through the SPIx peripheral.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  Data: Data to be transmitted.
 * @retval None.
 ******************************************************************************/
void SPI_SendData(SPI_Type *SPIx, uint16_t Data)
{
  /* Write in the TDR register the data to be sent */
  SPIx->TDR = (uint32_t)Data;
}

/**
 *******************************************************************************
 * @brief  Returns the most recent received data by the SPIx peripheral.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @retval uint16_t: The value of the received data.
 ******************************************************************************/
uint16_t SPI_ReceiveData(SPI_Type *SPIx)
{
  /* Check the SPI parameters */
  PARAM_ASSERT(IS_SPI_PERIPH(SPIx));

  /* Return the data in the RDR register */
  return (uint16_t)SPIx->RDR;
}

#if defined (DPM32M08x) || defined (DPM32M05x)
/**
 *******************************************************************************
 * @brief  Enables or disables the SPIx DMA interface.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_DMAReq: specifies the SPI DMA transfer request to be enabled or disabled.
 *               Any value of @ref SPI_DMAReqEnum.
 *                 @arg SPI_DMA_REQ_TX: Tx buffer DMA transfer request.
 *                 @arg SPI_DMA_REQ_RX: Rx buffer DMA transfer request.
 * @param  [in]  Newstate: New Newstate of the SPI DMA interface.
 *               Any value of @ref FunctionalState.
 *                 @arg DISABLE: disbale SPI DMA interface.
 *                 @arg ENABLE: enable SPI DMA interface.
 * @retval None.
 ******************************************************************************/
void SPI_DMACmd(SPI_Type *SPIx, SPI_DMAReqEnum SPI_DMAReq, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /* Enable the selected SPI DMA requests */
    SPIx->CR |= ((uint32_t)SPI_DMAReq);
  }
  else
  {
    /* Disable the selected SPI DMA requests */
    SPIx->CR &= ((uint32_t)~SPI_DMAReq);
  }
}

#endif  /* DPM32M08x || DPM32M05x */

/**
 *******************************************************************************
 * @brief  Enables or disables the specified SPI interrupts.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_IntType: Specifies the SPI interrupt source to be enabled or disabled.
 *               Any value of @ref SPI_IntTypeEnum.
 *                 @arg SPI_INT_TYPE_TXE: Tx FIFO empty interrupt.
 *                 @arg SPI_INT_TYPE_RXNE: Rx FIFO not empty interrupt.
 *                 @arg SPI_INT_TYPE_UDRR: Send invalid data interrupt in slave mode.
 *                 @arg SPI_INT_TYPE_ERR: SPI error interrupt.
 * @param  [in]  Newstate: New Newstate of the specified SPI interrupt.
 *               Any value of @ref FunctionalState.
 *                 @arg DISABLE: disbale SPI interrupt.
 *                 @arg ENABLE: enable SPI interrupt.
 * @retval None.
 ******************************************************************************/
void SPI_IntCmd(SPI_Type *SPIx, SPI_IntTypeEnum SPI_IntType, FunctionalState Newstate)
{
  if(DISABLE != Newstate)
  {
    /*Enable the selected SPI interrupt */
    SPIx->CR |= ((uint32_t)SPI_IntType);
  }
  else
  {
    /* Disable the selected SPI interrupt */
    SPIx->CR &= ((uint32_t)~SPI_IntType);
  }
}

/**
 *******************************************************************************
 * @brief  Get the specified SPI interrupt has enable or not.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_IntType: Specifies the SPI interrupt source to get enabled status.
 *               Any value of @ref SPI_IntTypeEnum.
 *                 @arg SPI_INT_TYPE_TXE: Tx FIFO empty interrupt.
 *                 @arg SPI_INT_TYPE_RXNE: Rx FIFO not empty interrupt.
 *                 @arg SPI_INT_TYPE_UDRR: Send invalid data interrupt in slave mode.
 *                 @arg SPI_INT_TYPE_ERR: SPI error interrupt.
 * @retval  FunctionalState: Interrupt enable bit has enable or not.
 *               Any value of @ref FunctionalState.
 *                 @arg DISABLE: Interrupt is disable.
 *                 @arg ENABLE: Interrupt is enable.
 ******************************************************************************/
FunctionalState SPI_GetIntCmdStatus(SPI_Type *SPIx, SPI_IntTypeEnum SPI_IntType)
{
  FunctionalState state = DISABLE;

  if(RESET != (SPIx->CR & SPI_IntType))
  {
    /* Interrupt is enable. */
    state = ENABLE;
  }
  else
  {
    /* Interrupt is disbale. */
    state = DISABLE;
  }

  /* Return the status of the interrupt enable bit. */
  return state;
}

/**
 *******************************************************************************
 * @brief  Checks whether the specified SPIx flag is set or not.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_IntFlag: Specifies the SPI flag to check.
 *               Any value of @ref SPI_IntFlagEnum.
 *                 @arg SPI_INT_FLAG_TXE: Tx FIFO is empty interrupt flag.
 *                 @arg SPI_INT_FLAG_RXNE: Rx FIFO is not empty interrupt flag.
 *                 @arg SPI_INT_FLAG_UDRR: Slave mode send data invalid interrupt flag.
 *                 @arg SPI_INT_FLAG_FAULT: NSS interference interrupt in slave mode.
 *                 @arg SPI_INT_FLAG_OVRR: RX_BUFF data unread overflow interrupt flag.
 * @retval FlagState: Interrupt flag has occurred or not.
 *               Any value of @ref FlagState.
 *                 @arg SET: Interrupt has occurred.
 *                 @arg RESET: Interrupt has not occurred.
 ******************************************************************************/
FlagState SPI_GetIntFlagStatus(SPI_Type *SPIx, SPI_IntFlagEnum SPI_IntFlag)
{
  FlagState state = RESET;

  if(RESET != (SPIx->ISR & SPI_IntFlag))
  {
    /* Interrupt has occurred. */
    state = SET;
  }
  else
  {
    /* Interrupt has not occurred. */
    state = RESET;
  }

  /* Return the status of the interrupt flag bit. */
  return state;
}

/**
 *******************************************************************************
 * @brief  Clears the SPIx interrupt flags.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @param  [in]  SPI_IntFlag: specifies the SPI flag to clear.
 *               Any value of @ref SPI_IntFlagEnum.
 *                 @arg SPI_INT_FLAG_TXE: Tx FIFO is empty interrupt flag.
 *                 @arg SPI_INT_FLAG_RXNE: Rx FIFO is not empty interrupt flag.
 *                 @arg SPI_INT_FLAG_UDRR: Slave mode send data invalid interrupt flag.
 *                 @arg SPI_INT_FLAG_FAULT: NSS interference interrupt in slave mode.
 *                 @arg SPI_INT_FLAG_OVRR: RX_BUFF data unread overflow interrupt flag.
 * @retval None.
 ******************************************************************************/
void SPI_ClearIntFlag(SPI_Type *SPIx, SPI_IntFlagEnum SPI_IntFlag)
{
  /* Clear interrupt flags.*/
  SPIx->ISR = SPI_IntFlag;
}

/**
 *******************************************************************************
 * @brief  Checks whether the specified SPIx is busy or not.
 * @param  [in]  SPIx: To select the SPIx peripheral, where x can be: 0 or 1.
 * @retval FlagState: Interrupt flag has occurred or not.
 *               Any value of @ref FlagState.
 *                 @arg SET: SPI is busy.
 *                 @arg RESET: SPI is not busy.
 ******************************************************************************/
FlagState SPI_GetBusyFlagStatus(SPI_Type *SPIx)
{
  FlagState state = RESET;

  if(RESET != (SPIx->ISR & SPI_ISR_BUSY_Msk))
  {
    state = SET;
  }
  else
  {
    state = RESET;
  }

  /* Return the status of the SPI busy flag bit. */
  return state;
}
