/**
 *******************************************************************************
 * @file    dpm32m0xx_tim.h
 *
 * @brief   Header file for TIM firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_TIM_H__
#define __DPM32M0XX_TIM_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"

/**
 ******************************************************************************
 * @brief TIM counter cycle enumeration.
 *****************************************************************************/
typedef enum
{
  TIM_COUNTER_CYCLE_PERIOD           = 0x00UL,                 /*!< Period cycle counter. */
  TIM_COUNTER_CYCLE_SINGLE           = TIM_CR_SINGLE_Msk       /*!< Single cycle counter. */
} TIM_CounterCycleEnum;

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 ******************************************************************************
 * @brief TIM trigger type enumeration.
 *****************************************************************************/
typedef enum
{
  TIM_TRIGGER_TYPE_AUTO_START        = 0x00000000UL,           /*!< After the current timer en is configured as 1, 
                                                                    the counting will start automatically. */
  TIM_TRIGGER_TYPE_SOURCE_START      = 0x00001000UL,           /*!< When the trigger counting source timer starts counting, 
                                                                    the current timer is automatically overloaded and starts counting. */
  TIM_TRIGGER_TYPE_SOURCE_COMPLETE   = 0x00002000UL            /*!< When the trigger counting source timer finishes counting, 
                                                                    the current timer will automatically reload and start counting. */
} TIM_TriggerTypeEnum;

/**
 ******************************************************************************
 * @brief TIM trigger source enumeration.
 *****************************************************************************/
typedef enum
{
  TIM_COUNTER_SOURCE_TIMER0          = 0x00000000UL,           /*!< Counter trigger count source timer0. */
  TIM_COUNTER_SOURCE_TIMER1          = 0x00000100UL,           /*!< Counter trigger count source timer1. */
#if defined (DPM32M036) 
  TIM_COUNTER_SOURCE_TIMER2          = 0x00000200UL            /*!< Counter trigger count source timer2. */
#endif /* DPM32M036 */
} TIM_TriggerSourceEnum;

#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 ******************************************************************************
 * @brief TIM initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint32_t                  TIM_ReloadValue;      /*!< Specifies the period value to be loaded
                                                       into the active Auto-Reload Register at the next update event.
                                                       This parameter can be a number between 0x00000000 and 0xFFFFFFFF.*/
  TIM_CounterCycleEnum      TIM_CounterCycle;     /*!< Specifies the counter cycle.
                                                       This parameter can be a value of @ref TIM_CounterCycleEnum */
  FunctionalState           TIM_IntState;         /*!< Specifies the interrupt enable for the TIMx.
                                                       This parameter can be: ENABLE or DISABLE. */
#if defined (DPM32M08x) || defined (DPM32M05x)
  FunctionalState           TIM_DMAState;         /*!< Specifies the DMA enable for TIMx.
                                                       This parameter can be: ENABLE or DISABLE. */
#endif /* DPM32M08x || DPM32M05x */
} TIM_InitTypeStruct;

/* TIM Param Check. */
#if defined (DPM32M08x) || defined (DPM32M05x)
#define IS_TIM_PERIPH(x) (((x) == TIM0) || \
                          ((x) == TIM1) || \
                          ((x) == TIM2) || \
                          ((x) == TIM3) || \
                          ((x) == TIM4) || \
                          ((x) == TIM5))
#elif defined (DPM32M03x) || defined (DPM32M036)
#define IS_TIM_PERIPH(x) (((x) == TIM0) || \
                          ((x) == TIM1) || \
                          ((x) == TIM2))
#elif defined (DPM32M030) || defined (DPM32M015)
#define IS_TIM_PERIPH(x) (((x) == TIM0) || \
                          ((x) == TIM1))
#endif /* DPM32M08x || DPM32M05x */

#define IS_TIM_COUNTER_MODE(x) (((x) == TIM_COUNTER_CYCLE_PERIOD) || \
                                ((x) == TIM_COUNTER_CYCLE_SINGLE))



/* Deinitializes and initialization and cmd functions. */
void                    TIM_DeInit(TIM_Type* TIMx);
void                    TIM_Init(TIM_Type* TIMx, TIM_InitTypeStruct* TIM_InitType);
void                    TIM_StructInit(TIM_InitTypeStruct* TIM_InitType);
void                    TIM_Cmd(TIM_Type* TIMx, FunctionalState NewState);

/* TIMx reload counter functions. */
void                    TIM_SetAutoReload(TIM_Type* TIMx, uint32_t TIM_Value);
void                    TIM_SetCounter(TIM_Type* TIMx, uint32_t TIM_Value);
uint32_t                TIM_GetAutoReload(TIM_Type* TIMx);
uint32_t                TIM_GetCounter(TIM_Type* TIMx);

/* TIMx counter mode configuration functions. */
void                    TIM_CounterCycleConfig(TIM_Type* TIMx, TIM_CounterCycleEnum TIM_Cycle);

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/* TIMx cascade trigger functions. */
void                    TIM_TriggerTypeConfig(TIM_Type* TIMx, TIM_TriggerTypeEnum TIM_TriggerType);
void                    TIM_TriggerSourceConfig(TIM_Type* TIMx, TIM_TriggerSourceEnum TIM_TriggerSource);
/* TIMx busy functions. */
FlagState               TIM_GetBusy(TIM_Type* TIMx);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#if defined (DPM32M08x) || defined (DPM32M05x)
/* Enable or disable TIMx DMA functions. */
void                    TIM_DMACmd(TIM_Type* TIMx, FunctionalState NewState);
#endif  /* DPM32M08x || DPM32M05x */

/* TIMx interrupt related functions. */
void                    TIM_IntCmd(TIM_Type* TIMx, FunctionalState NewState);
FunctionalState         TIM_GetIntCmdStatus(TIM_Type* TIMx);
FlagState               TIM_GetIntFlagStatus(TIM_Type* TIMx);
void                    TIM_ClearIntFlag(TIM_Type* TIMx);


#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_TIM_H__ */
