/**
 *******************************************************************************
 * @file    dpm32m0xx_cct.c
 *
 * @brief   Source file for CCT firmware driver.
 *          This file provides firmware functions to manage the following
 *          functionalities of the Capture Compare Timer (CCT) peripheral:
 *            + TimeBase management
 *            + Output Compare management
 *            + Input Capture management
 *            + Interrupts, DMA and flags management
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 * @verbatim
 ===============================================================================
                       ##### How to use this driver #####
 ===============================================================================
    [..]
      (#) Enable the CCT controller interface clock using
          RCC_APBPeriphClockCmd(RCC_APB_PERIPH_CCT0, ENABLE) and
          RCC_APBPeriphClockCmd(RCC_APB_PERIPH_CCT1, ENABLE)

      (#) CCT pins configuration:
        (+) Enable the clock for the CCT GPIOs using the following function:
            RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE).
        (+) Configure these CCT pins in alternate function mode by calling
            the function GPIO_Init().
        (+) Connect the involved pins to alternate function using the
            following function GPIO_AltFuncConfig().

      (#) Configure the CCT reload value, clock division and counter cycle
          using the CCT_TimeBaseInit() function.

      (#) Activate the CCT peripheral using CCT_Cmd() function.

      (#) CCT output compare configuration:
        (+) Configure the output compare pulse value and polarity using
            CCT_OC0Init() and CCT_OC1Init() function.
        (+) Modify output compare values using CCT_SetCompare0() and
            CCT_SetCompare1() function.

      (#) CCT input capture configuration:
        (+) Configure the input capture singal source, ACMP channel, filter,
            action and polarity using CCT_IC0Init() and CCT_IC1Init() function.
        (+) Get capture values using CCT_GetCapture0() and CCT_GetCapture1()
            function.
        (+) Get capture status using CCT_GetFlagStatus() function.
        (+) Clear capture status using CCT_ClearFlag() function.

      (#) Enable the channel functions using the CCT_CH0Cmd() and CCT_CH1Cmd
          function.

      (#) Enable the DMA using the function CCT_DMACmd() if you need to use DMA
          functions.

      (#) CCT interrupt configuration:
        (+) To activate the CCT interrupt, use CCT_IntCmd() functions.
        (+) Check on CCT interrupt enable flags using the function
            CCT_GetIntCmdStatus().
        (+) Check on CCT interrupt occur flags using the function
            CCT_GetIntFlagStatus().
        (+) Clear CCT interrupt flags using the function CCT_ClearIntFlag().

 * @endverbatim
 *******************************************************************************/

#include "dpm32m0xx_cct.h"

/**
 *******************************************************************************
 * @brief   Deinitializes the CCTx peripheral registers to their default reset values.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @retval  None.
 ******************************************************************************/
void CCT_DeInit(CCT_Type* CCTx)
{
  /* Reset CCTx control register. */
  CCTx->CR = (uint32_t)0x00000000UL;

  /* Reset CCTx counter register. */
  CCTx->ARR = (uint32_t)0x00000000UL;

  /* Clear CCTx all flags. */
  CCTx->SR = (uint32_t)0x0000037FUL;

  /* Reset CCTx compare control register. */
  CCTx->CMP_CFG = (uint32_t)0x00000000UL;

  /* Reset CCTx capture control register. */
  CCTx->CAP_CFG = (uint32_t)0x00000000UL;

  /* Reset CCTx channel0 value register. */
  CCTx->CH0_VALUE = (uint32_t)0x00000000UL;

  /* Reset CCTx channel1 value register. */
  CCTx->CH1_VALUE = (uint32_t)0x00000000UL;

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  /* Reset CCTx channel2 value register. */
  CCTx->CH2_VALUE = (uint32_t)0x00000000UL;
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
}

/**
 *******************************************************************************
 * @brief   Initializes the CCTx peripheral according to the specified parameters
 *          in the CCT_TimeBaseInitType.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_TimeBaseInitType: Structure pointer of CCTx configuration.
 * @retval  None.
 ******************************************************************************/
void CCT_TimeBaseInit(CCT_Type* CCTx, CCT_TimeBaseInitTypeStruct* CCT_TimeBaseInitType)
{
  uint32_t tmpReg = 0UL;
  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_CLOCK_DIV(CCT_TimeBaseInitType->CCT_ClockDivision));
  PARAM_ASSERT(IS_CCT_COUNTER_MODE(CCT_TimeBaseInitType->CCT_CounterCycle));

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx CLK_DIV[2:0],SINGLE bits. */
  tmpReg &= ~(CCT_CR_CLK_DIV_Msk | CCT_CR_SINGLE_Msk);

  /* Set CCTx CLK_DIV[2:0] bits according to CCT_TimeBaseInitType. */
  tmpReg |= (uint32_t)(CCT_TimeBaseInitType->CCT_ClockDivision);

  /* Set CCTx mode bit according to CCT_TimeBaseInitType. */
  tmpReg |= (uint32_t)(CCT_TimeBaseInitType->CCT_CounterCycle);

  /* Store the new value. */
  CCTx->CR = tmpReg;

  /* Sets the CCTx auto reload register value. */
  CCTx->ARR = CCT_TimeBaseInitType->CCT_ReloadValue;
}

/**
 *******************************************************************************
 * @brief   Initialize the CCT_TimeBaseInitType with default parameters.
 * @param   [in]  CCT_TimeBaseInitType: Pointer to a CCT_TimeBaseInitTypeStruct
 *                structure which will be initialized.
 * @retval  None.
 ******************************************************************************/
void CCT_TimeBaseStructInit(CCT_TimeBaseInitTypeStruct* CCT_TimeBaseInitType)
{
  CCT_TimeBaseInitType->CCT_ReloadValue = 0xFFFF;
  CCT_TimeBaseInitType->CCT_ClockDivision = CCT_CLK_DIV1;
  CCT_TimeBaseInitType->CCT_CounterCycle = CCT_COUNTER_CYCLE_PERIOD;
}

/**
 *******************************************************************************
 * @brief   Enables or disables the specified CCT peripheral.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void CCT_Cmd(CCT_Type* CCTx, FunctionalState NewState)
{
  if(DISABLE != NewState)
  {
    /* Enable CCTx peripheral. */
    CCTx->CR |= CCT_CR_EN_Msk;
  }
  else
  {
    /* Disable CCTx peripheral. */
    CCTx->CR &= ~CCT_CR_EN_Msk;
  }
}

/**
 *******************************************************************************
 * @brief   Sets the CCTx auto reload register value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Value: Specifies the auto reload register new value.
 * @retval  None.
 ******************************************************************************/
void CCT_SetAutoReload(CCT_Type* CCTx, uint16_t CCT_Value)
{
  /* Set the auto reload register value. */
  CCTx->ARR = (uint32_t)CCT_Value;
}

/**
 *******************************************************************************
 * @brief   Gets the CCTx auto reload register value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @retval  uint16_t: Auto reload register value.
 ******************************************************************************/
uint16_t CCT_GetAutoReload(CCT_Type* CCTx)
{
  /* Get the auto reload register value. */
  return (uint16_t)CCTx->ARR;
}

/**
 *******************************************************************************
 * @brief   Gets the CCTx counter value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @retval  uint16_t: Counter register value.
 ******************************************************************************/
uint16_t CCT_GetCounter(CCT_Type* CCTx)
{
  /* Get the counter register value. */
  return (uint16_t)CCTx->CNT;
}

/**
 *******************************************************************************
 * @brief   Configuration the CCTx cycle counter.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Cycle: Specifies CCTx counter cycle.
 *                This parameter can be a value of @ref CCT_CounterCycleEnum.
 *                  @arg CCT_COUNTER_CYCLE_PERIOD: Period cycle counter.
 *                  @arg CCT_COUNTER_CYCLE_SINGLE: Single cycle counter.
 * @retval  None.
 ******************************************************************************/
void CCT_CounterCycleConfig(CCT_Type* CCTx, CCT_CounterCycleEnum CCT_Cycle)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx mode bit. */
  tmpReg &= ~CCT_CR_SINGLE_Msk;

  /* Set CCTx mode bit according to mode. */
  tmpReg |= (uint32_t)(CCT_Cycle);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration the CCTx clock division.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Div: Specifies CCTx clock division.
 *                This parameter can be a value of @ref CCT_ClockDivEnum.
 *                  @arg CCT_CLK_DIV1: CCT clock is pclk.
 *                  @arg CCT_CLK_DIV2: CCT clock is pclk/2.
 *                  @arg CCT_CLK_DIV4: CCT clock is pclk/4.
 *                  @arg CCT_CLK_DIV8: CCT clock is pclk/8.
 *                  @arg CCT_CLK_DIV16: CCT clock is pclk/16.
 *                  @arg CCT_CLK_DIV32: CCT clock is pclk/32.
 *                  @arg CCT_CLK_DIV64: CCT clock is pclk/64.
 *                  @arg CCT_CLK_DIV128: CCT clock is pclk/128.
 * @retval  None.
 ******************************************************************************/
void CCT_ClockDivConfig(CCT_Type* CCTx, CCT_ClockDivEnum CCT_Div)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx CLK_DIV[2:0] bits. */
  tmpReg &= ~CCT_CR_CLK_DIV_Msk;

  /* Set CCTx CLK_DIV[2:0] bits according to CCT_Div. */
  tmpReg |= (uint32_t)(CCT_Div);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Initializes the CCTx output compare channel0 according to the specified
 *          parameters in the CCT_OCInitType.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_OCInitType: Pointer to a CCT_OCInitTypeStruct structure that contains
 *                the configuration information for the specified CCT output compare.
 * @retval  None.
 ******************************************************************************/
void CCT_OC0Init(CCT_Type* CCTx, CCT_OCInitTypeStruct* CCT_OCInitType)
{
  uint32_t tmpReg = 0UL;

  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_OC_POLARITY(CCT_OCInitType->CCT_OCPolarity));

  /* Set CCTx CH0_MODE bit as output compare mode. */
  CCTx->CR |= (uint32_t)(CCT_CR_CH0_MODE_Msk);

  /* Read the value of the CCTx compare control register. */
  tmpReg = CCTx->CMP_CFG;

  if(CCT_OC_POLARITY_HIGH == CCT_OCInitType->CCT_OCPolarity)
  {
    /* Set CCTx CH0_INIT_O bit. */
    tmpReg |= (uint32_t)CCT_CMP_CFG_CH0_INIT_O_Msk;
  }
  else
  {
    /* Clear CCTx CH0_INIT_O bit. */
    tmpReg &= ~(uint32_t)CCT_CMP_CFG_CH0_INIT_O_Msk;
  }

  /* Store the new value. */
  CCTx->CMP_CFG = tmpReg;

  /* Set CCTx CH0_VALUE[15:0] bits according to CCT_OCInitType. */
  CCTx->CH0_VALUE = CCT_OCInitType->CCT_OCPulse;
}

/**
 *******************************************************************************
 * @brief   Initializes the CCTx output compare channel1 according to the specified
 *          parameters in the CCT_OCInitType.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_OCInitType: Pointer to a CCT_OCInitTypeStruct structure that contains
 *                the configuration information for the specified CCT output compare.
 * @retval  None.
 ******************************************************************************/
void CCT_OC1Init(CCT_Type* CCTx, CCT_OCInitTypeStruct* CCT_OCInitType)
{
  uint32_t tmpReg = 0UL;

  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_OC_POLARITY(CCT_OCInitType->CCT_OCPolarity));

  /* Set CCTx CH1_MODE bit as output compare mode. */
  CCTx->CR |= (uint32_t)(CCT_CR_CH1_MODE_Msk);

  /* Read the value of the CCTx compare control register. */
  tmpReg = CCTx->CMP_CFG;

  if(CCT_OC_POLARITY_HIGH == CCT_OCInitType->CCT_OCPolarity)
  {
    /* Set CCTx CH1_INIT_O bit. */
    tmpReg |= (uint32_t)CCT_CMP_CFG_CH1_INIT_O_Msk;
  }
  else
  {
    /* Clear CCTx CH1_INIT_O bit. */
    tmpReg &= ~(uint32_t)CCT_CMP_CFG_CH1_INIT_O_Msk;
  }

  /* Store the new value. */
  CCTx->CMP_CFG = tmpReg;

  /* Set CCTx CH1_VALUE[15:0] bits according to CCT_OCInitType. */
  CCTx->CH1_VALUE = CCT_OCInitType->CCT_OCPulse;
}

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 *******************************************************************************
 * @brief   Initializes the CCTx output compare channel2 according to the specified
 *          parameters in the CCT_OCInitType.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_OCInitType: Pointer to a CCT_OCInitTypeStruct structure that contains
 *                the configuration information for the specified CCT output compare.
 * @retval  None.
 ******************************************************************************/
void CCT_OC2Init(CCT_Type* CCTx, CCT_OCInitTypeStruct* CCT_OCInitType)
{
  uint32_t tmpReg = 0UL;

  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_OC_POLARITY(CCT_OCInitType->CCT_OCPolarity));

  /* Set CCTx CH2_MODE bit as output compare mode. */
  CCTx->CR |= (uint32_t)(CCT_CR_CH2_MODE_Msk);

  /* Read the value of the CCTx compare control register. */
  tmpReg = CCTx->CMP_CFG;

  if(CCT_OC_POLARITY_HIGH == CCT_OCInitType->CCT_OCPolarity)
  {
    /* Set CCTx CH2_INIT_O bit. */
    tmpReg |= (uint32_t)CCT_CMP_CFG_CH2_INIT_O_Msk;
  }
  else
  {
    /* Clear CCTx CH2_INIT_O bit. */
    tmpReg &= ~(uint32_t)CCT_CMP_CFG_CH2_INIT_O_Msk;
  }

  /* Store the new value. */
  CCTx->CMP_CFG = tmpReg;

  /* Set CCTx CH2_VALUE[15:0] bits according to CCT_OCInitType. */
  CCTx->CH2_VALUE = CCT_OCInitType->CCT_OCPulse;
}
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
/**
 *******************************************************************************
 * @brief   Initialize the CCT_OCInitType with default parameters.
 * @param   [in]  CCT_OCInitType: Pointer to a CCT_OCInitTypeStruct structure
 *                which will be initialized.
 * @retval  None.
 ******************************************************************************/
void CCT_OCStructInit(CCT_OCInitTypeStruct* CCT_OCInitType)
{
  CCT_OCInitType->CCT_OCPulse = 0xFFFF;
  CCT_OCInitType->CCT_OCPolarity = CCT_OC_POLARITY_HIGH;
}

/**
 *******************************************************************************
 * @brief   Sets the CCTx channel0 compare register value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Value: Specifies the channel0 compare register new value.
 * @retval  None.
 ******************************************************************************/
void CCT_SetCompare0(CCT_Type* CCTx, uint16_t CCT_Value)
{
  /* Set the channel0 compare register value */
  CCTx->CH0_VALUE = (uint32_t)CCT_Value;
}

/**
 *******************************************************************************
 * @brief   Sets the CCTx channel1 compare Register value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Value: Specifies the channel1 compare register new value.
 * @retval  None.
 ******************************************************************************/
void CCT_SetCompare1(CCT_Type* CCTx, uint16_t CCT_Value)
{
  /* Set the channel1 compare register value */
  CCTx->CH1_VALUE = (uint32_t)CCT_Value;
}

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 *******************************************************************************
 * @brief   Sets the CCTx channel2 compare Register value.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Value: Specifies the channel2 compare register new value.
 * @retval  None.
 ******************************************************************************/
void CCT_SetCompare2(CCT_Type* CCTx, uint16_t CCT_Value)
{
  /* Set the channel2 compare register value */
  CCTx->CH2_VALUE = (uint32_t)CCT_Value;
}
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 *******************************************************************************
 * @brief   Configures the CCTx output compare channel0 polarity.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Polarity: Specifies the OC0 polarity.
 *                This parameter can be a value of @ref CCT_OCPolarityEnum.
 *                  @arg CCT_OC_POLARITY_HIGH: Output Compare active high.
 *                  @arg CCT_OC_POLARITY_LOW: Output Compare active low.
 * @retval  None.
 ******************************************************************************/
void CCT_OC0PolarityConfig(CCT_Type* CCTx, CCT_OCPolarityEnum CCT_Polarity)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx compare control register. */
  tmpReg = CCTx->CMP_CFG;

  if(CCT_OC_POLARITY_HIGH == CCT_Polarity)
  {
    /* Set CCTx CH0_INIT_O bit. */
    tmpReg |= (uint32_t)CCT_CMP_CFG_CH0_INIT_O_Msk;
  }
  else
  {
    /* Clear CCTx CH0_INIT_O bit. */
    tmpReg &= ~(uint32_t)CCT_CMP_CFG_CH0_INIT_O_Msk;
  }

  /* Store the new value. */
  CCTx->CMP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configures the CCTx output compare channel1 polarity.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Polarity: Specifies the OC1 polarity.
 *                This parameter can be a value of @ref CCT_OCPolarityEnum.
 *                  @arg CCT_OC_POLARITY_HIGH: Output Compare active high.
 *                  @arg CCT_OC_POLARITY_LOW: Output Compare active low.
 * @retval  None.
 ******************************************************************************/
void CCT_OC1PolarityConfig(CCT_Type* CCTx, CCT_OCPolarityEnum CCT_Polarity)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx compare control register. */
  tmpReg = CCTx->CMP_CFG;

  if(CCT_OC_POLARITY_HIGH == CCT_Polarity)
  {
    /* Set CCTx CH1_INIT_O bit. */
    tmpReg |= (uint32_t)CCT_CMP_CFG_CH1_INIT_O_Msk;
  }
  else
  {
    /* Clear CCTx CH1_INIT_O bit. */
    tmpReg &= ~(uint32_t)CCT_CMP_CFG_CH1_INIT_O_Msk;
  }

  /* Store the new value. */
  CCTx->CMP_CFG = tmpReg;
}

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 *******************************************************************************
 * @brief   Configures the CCTx output compare channel2 polarity.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Polarity: Specifies the OC2 polarity.
 *                This parameter can be a value of @ref CCT_OCPolarityEnum.
 *                  @arg CCT_OC_POLARITY_HIGH: Output Compare active high.
 *                  @arg CCT_OC_POLARITY_LOW: Output Compare active low.
 * @retval  None.
 ******************************************************************************/
void CCT_OC2PolarityConfig(CCT_Type* CCTx, CCT_OCPolarityEnum CCT_Polarity)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx compare control register. */
  tmpReg = CCTx->CMP_CFG;

  if(CCT_OC_POLARITY_HIGH == CCT_Polarity)
  {
    /* Set CCTx CH2_INIT_O bit. */
    tmpReg |= (uint32_t)CCT_CMP_CFG_CH2_INIT_O_Msk;
  }
  else
  {
    /* Clear CCTx CH2_INIT_O bit. */
    tmpReg &= ~(uint32_t)CCT_CMP_CFG_CH2_INIT_O_Msk;
  }

  /* Store the new value. */
  CCTx->CMP_CFG = tmpReg;
}
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 *******************************************************************************
 * @brief   Initialize the CCT_ICInitType with default parameters.
 * @param   [in]  CCT_ICInitType Pointer to a CCT_ICInitTypeStruct structure
 *                which will be initialized.
 * @retval  None.
 ******************************************************************************/
void CCT_ICStructInit(CCT_ICInitTypeStruct* CCT_ICInitType)
{
  CCT_ICInitType->CCT_ICSingalSel = CCT_IC_SINGAL_SOURCE_PIN;
#if defined (DPM32M015)
  CCT_ICInitType->CCT_ICACMPChl = CCT_IC_ACMP1;
#else /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  CCT_ICInitType->CCT_ICACMPChl = CCT_IC_ACMP0;
#endif /* DPM32M015 */
  CCT_ICInitType->CCT_ICFiltSampDiv = CCT_IC_FILT_SAMPLE_DIV1;
  CCT_ICInitType->CCT_ICFiltLen = CCT_IC_FILT_LEN1;
  CCT_ICInitType->CCT_ICAction = CCT_IC_ACTION_NO;
  CCT_ICInitType->CCT_ICPolarity = CCT_IC_POLARITY_RISING;
}

/**
 *******************************************************************************
 * @brief   Get the specified CCT flag is set or not.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_IntFlag: Specifies the CCT flag.
 *                This parameter can be a value of @ref CCT_FlagEnum.
 *                  @arg CCT_FLAG_CH0_RISING_CAPTURE: Channel0 capture rising flag.
 *                  @arg CCT_FLAG_CH0_FALLING_CAPTURE: Channel0 capture falling flag.
 *                  @arg CCT_FLAG_CH1_RISING_CAPTURE: Channel1 capture rising flag.
 *                  @arg CCT_FLAG_CH1_FALLING_CAPTURE: Channel1 capture falling flag.
 *                  @arg CCT_FLAG_CH2_RISING_CAPTURE: Channel2 capture rising flag.
 *                  @arg CCT_FLAG_CH2_FALLING_CAPTURE: Channel2 capture falling flag.
 * @retval  FlagState: The new state of CCT_IntFlag (SET or RESET).
 ******************************************************************************/
FlagState CCT_GetFlagStatus(CCT_Type* CCTx, CCT_FlagEnum CCT_Flag)
{
  FlagState state = RESET;

  /* Get the status of the Interrupt */
  if(RESET != (CCTx->SR & ((uint32_t)CCT_Flag)))
  {
    state = SET;
  }
  else
  {
    state = RESET;
  }

  /* Return the status of the interrupt flag bit. */
  return state;
}

/**
 *******************************************************************************
 * @brief   Get the specified CCT flag is set or not.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_IntFlag: Specifies the CCT flag.
 *                This parameter can be a value of @ref CCT_FlagEnum.
 *                  @arg CCT_FLAG_CH0_RISING_CAPTURE: Channel0 capture rising flag.
 *                  @arg CCT_FLAG_CH0_FALLING_CAPTURE: Channel0 capture falling flag.
 *                  @arg CCT_FLAG_CH1_RISING_CAPTURE: Channel1 capture rising flag.
 *                  @arg CCT_FLAG_CH1_FALLING_CAPTURE: Channel1 capture falling flag.
 *                  @arg CCT_FLAG_CH2_RISING_CAPTURE: Channel2 capture rising flag.
 *                  @arg CCT_FLAG_CH2_FALLING_CAPTURE: Channel2 capture falling flag.
 * @retval  None.
 ******************************************************************************/
void CCT_ClearFlag(CCT_Type* CCTx, CCT_FlagEnum CCT_Flag)
{
  /* Clear interrupt flags.*/
  CCTx->SR = ((uint32_t)CCT_Flag);
}

/**
 *******************************************************************************
 * @brief   Initializes the CCTx input capture channel0 according to the specified
 *          parameters in the CCT_ICInitType.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_ICInitType: Pointer to a CCT_ICInitTypeStruct structure that contains
 *                the configuration information for the specified CCT input capture.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0Init(CCT_Type* CCTx, CCT_ICInitTypeStruct* CCT_ICInitType)
{
  uint32_t tmpReg = 0, tmpReg1 = 0;
  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_IC_SINGLE_SOURCE(CCT_ICInitType->CCT_ICSingalSel));
  PARAM_ASSERT(IS_CCT_IC_ACMP_CHANNEL(CCT_ICInitType->CCT_ICACMPChl));
  PARAM_ASSERT(IS_CCT_IC_FILT_SAMPLE_DIV(CCT_ICInitType->CCT_ICFiltSampDiv));
  PARAM_ASSERT(IS_CCT_IC_FILT_LEN(CCT_ICInitType->CCT_ICFiltLen));
  PARAM_ASSERT(IS_CCT_IC_ACTION(CCT_ICInitType->CCT_ICAction));
  PARAM_ASSERT(IS_CCT_IC_POLARITY(CCT_ICInitType->CCT_ICPolarity));

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN0_ACMP_SEL[1:0],IN0_SEL,CH0_MODE bits. */
  tmpReg &= ~(CCT_CR_IN0_ACMP_SEL_Msk | CCT_CR_IN0_SEL_Msk | CCT_CR_CH0_MODE_Msk);

  /* Set CCTx IN0_ACMP_SEL[1:0] bits according to CCT_ICInitType. */
  tmpReg |= (uint32_t)(CCT_ICInitType->CCT_ICACMPChl << CCT_CR_IN0_ACMP_SEL_Pos);

  /* Set CCTx IN0_SEL bit according to CCT_ICInitType. */
  tmpReg |= (uint32_t)(CCT_ICInitType->CCT_ICSingalSel << CCT_CR_IN0_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;

  /* Read the value of the CCTx input capture control register. */
  tmpReg1 = CCTx->CAP_CFG;

  /* Clear CCTx CH0_FLT_SAMPLE[1:0],CH0_FLT_LEN[1:0],CH0_CNT_CLR,CH0_EDGE[1:0] bits. */
  tmpReg1 &= ~(CCT_CAP_CFG_CH0_FLT_SAMPLE_Msk | CCT_CAP_CFG_CH0_FLT_LEN_Msk | CCT_CAP_CFG_CH0_CNT_CLR_Msk \
               | CCT_CAP_CFG_CH0_EDGE_Msk);

  /* Set CCTx CH0_FLT_SAMPLE[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICFiltSampDiv << CCT_CAP_CFG_CH0_FLT_SAMPLE_Pos);

  /* Set CCTx CH0_FLT_LEN[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICFiltLen << CCT_CAP_CFG_CH0_FLT_LEN_Pos);

  /* Set CCTx CH0_CNT_CLR bit according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICAction << CCT_CAP_CFG_CH0_CNT_CLR_Pos);

  /* Set CCTx CH0_EDGE[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICPolarity << CCT_CAP_CFG_CH0_EDGE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg1;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel0 single source.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Source: Input capture singal sourcce.
 *                This parameter can be a value of @ref CCT_ICSingalSourceEnum.
 *                  @arg CCT_IC_SINGAL_SOURCE_PIN: Used pin input as source.
 *                  @arg CCT_IC_SINGAL_SOURCE_ACMP: Used ACMP as source.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0SingleSourceConfig(CCT_Type* CCTx, CCT_ICSingalSourceEnum CCT_Source)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN0_SEL bit. */
  tmpReg &= ~CCT_CR_IN0_SEL_Msk;

  /* Set CCTx IN0_SEL bit according to source. */
  tmpReg |= (uint32_t)(CCT_Source << CCT_CR_IN0_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel0 ACMP channel.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Chl: Input capture analog channel.
 *                This parameter can be a value of @ref CCT_ICACMPChannelEnum.
 *                DPM32M08x|DPM32M05x|DPM32M03x:
 *                  @arg CCT_IC_ACMP0: Used ACMP0 as input capture source.
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 *                  @arg CCT_IC_ACMP2: Used ACMP2 as input capture source.
 *                  @arg CCT_IC_ACMP3: Used ACMP3 as input capture source.
 *                DPM32M036|DPM32M030:
 *                  @arg CCT_IC_ACMP0: Used ACMP0 as input capture source.
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 *                DPM32M015:
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0ACMPChannelConfig(CCT_Type* CCTx, CCT_ICACMPChannelEnum CCT_Chl)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN0_ACMP_SEL[1:0] bits. */
  tmpReg &= ~CCT_CR_IN0_ACMP_SEL_Msk;

  /* Set CCTx IN0_ACMP_SEL[1:0] bits according to chl. */
  tmpReg |= (uint32_t)(CCT_Chl << CCT_CR_IN0_ACMP_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel0 filter sample division.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Div: Input capture channel filter sample division.
 *                This parameter can be a value of @ref CCT_ICFiltSampleDivEnum.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV1: Filter sample 1 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV4: Filter sample 4 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV16: Filter sample 16 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV32: Filter sample 32 division.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0FltSampDivConfig(CCT_Type* CCTx, CCT_ICFiltSampleDivEnum CCT_Div)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH0_FLT_SAMPLE[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH0_FLT_SAMPLE_Msk;

  /* Set CCTx CH0_FLT_SAMPLE[1:0] bits according to div. */
  tmpReg |= (uint32_t)(CCT_Div << CCT_CAP_CFG_CH0_FLT_SAMPLE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel0 filter length.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Len: Input capture channel filter length.
 *                This parameter can be a value of @ref CCT_ICFiltLenEnum.
 *                  @arg CCT_IC_FILT_LEN1: Filter 1 length.
 *                  @arg CCT_IC_FILT_LEN8: Filter 8 length.
 *                  @arg CCT_IC_FILT_LEN16: Filter 16 length.
 *                  @arg CCT_IC_FILT_LEN32: Filter 32 length.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0FltLenConfig(CCT_Type* CCTx, CCT_ICFiltLenEnum CCT_Len)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH0_FLT_LEN[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH0_FLT_LEN_Msk;

  /* Set CCTx CH0_FLT_LEN[1:0] bits according to len. */
  tmpReg |= (uint32_t)(CCT_Len << CCT_CAP_CFG_CH0_FLT_LEN_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel0 action after capture.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Act: Input capture action after capture.
 *                This parameter can be a value of @ref CCT_ICActionEnum.
 *                  @arg CCT_IC_ACTION_NO: No action after capture.
 *                  @arg CCT_IC_ACTION_CLEAR_CNT: Clear counter register after capture.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0ActionConfig(CCT_Type* CCTx, CCT_ICActionEnum CCT_Act)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH0_CNT_CLR bit. */
  tmpReg &= ~CCT_CAP_CFG_CH0_CNT_CLR_Msk;

  /* Set CCTx CH0_CNT_CLR bit according to act. */
  tmpReg |= (uint32_t)(CCT_Act << CCT_CAP_CFG_CH0_CNT_CLR_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel0 capture polarity.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Pol: Input capture polarity.
 *                This parameter can be a value of @ref CCT_ICPolarityEnum.
 *                  @arg CCT_IC_POLARITY_RISING: Capture rising edge.
 *                  @arg CCT_IC_POLARITY_FALLING: Capture falling edge.
 *                  @arg CCT_IC_POLARITY_RISINGFALLING: Capture rising and falling edge.
 * @retval  None.
 ******************************************************************************/
void CCT_IC0PolarityConfig(CCT_Type* CCTx, CCT_ICPolarityEnum CCT_Pol)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH0_EDGE[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH0_EDGE_Msk;

  /* Set CCTx CH0_EDGE[1:0] bits according to pol. */
  tmpReg |= (uint32_t)(CCT_Pol << CCT_CAP_CFG_CH0_EDGE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Get CH0 input capture value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @retval  uint16_t: CH0 input capture value.
 ******************************************************************************/
uint16_t CCT_GetCapture0(CCT_Type* CCTx)
{
  return (uint16_t)CCTx->CH0_VALUE;
}

/**
 *******************************************************************************
 * @brief   Initializes the CCTx input capture channel1 according to the specified
 *          parameters in the CCT_ICInitType.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_ICInitType: Pointer to a CCT_ICInitTypeStruct structure that contains
 *                the configuration information for the specified CCT input capture.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1Init(CCT_Type* CCTx, CCT_ICInitTypeStruct* CCT_ICInitType)
{
  uint32_t tmpReg = 0, tmpReg1 = 0;
  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_IC_SINGLE_SOURCE(CCT_ICInitType->CCT_ICSingalSel));
  PARAM_ASSERT(IS_CCT_IC_ACMP_CHANNEL(CCT_ICInitType->CCT_ICACMPChl));
  PARAM_ASSERT(IS_CCT_IC_FILT_SAMPLE_DIV(CCT_ICInitType->CCT_ICFiltSampDiv));
  PARAM_ASSERT(IS_CCT_IC_FILT_LEN(CCT_ICInitType->CCT_ICFiltLen));
  PARAM_ASSERT(IS_CCT_IC_ACTION(CCT_ICInitType->CCT_ICAction));
  PARAM_ASSERT(IS_CCT_IC_POLARITY(CCT_ICInitType->CCT_ICPolarity));

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN1_ACMP_SEL[1:0],IN1_SEL,CH1_MODE bits. */
  tmpReg &= ~(CCT_CR_IN1_ACMP_SEL_Msk | CCT_CR_IN1_SEL_Msk | CCT_CR_CH1_MODE_Msk);

  /* Set CCTx IN1_ACMP_SEL[1:0] bits according to CCT_ICInitType. */
  tmpReg |= (uint32_t)(CCT_ICInitType->CCT_ICACMPChl << CCT_CR_IN1_ACMP_SEL_Pos);

  /* Set CCTx IN1_SEL bit according to CCT_ICInitType. */
  tmpReg |= (uint32_t)(CCT_ICInitType->CCT_ICSingalSel << CCT_CR_IN1_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;

  /* Read the value of the CCTx input capture control register. */
  tmpReg1 = CCTx->CAP_CFG;

  /* Clear CCTx CH1_FLT_SAMPLE[1:0],CH1_FLT_LEN[1:0],CH1_CNT_CLR,CH1_EDGE[1:0] bits. */
  tmpReg1 &= ~(CCT_CAP_CFG_CH1_FLT_SAMPLE_Msk | CCT_CAP_CFG_CH1_FLT_LEN_Msk \
               | CCT_CAP_CFG_CH1_CNT_CLR_Msk | CCT_CAP_CFG_CH1_EDGE_Msk);

  /* Set CCTx CH1_FLT_SAMPLE[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICFiltSampDiv << CCT_CAP_CFG_CH1_FLT_SAMPLE_Pos);

  /* Set CCTx CH1_FLT_LEN[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICFiltLen << CCT_CAP_CFG_CH1_FLT_LEN_Pos);

  /* Set CCTx CH1_CNT_CLR bit according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICAction << CCT_CAP_CFG_CH1_CNT_CLR_Pos);

  /* Set CCTx CH1_EDGE[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICPolarity << CCT_CAP_CFG_CH1_EDGE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg1;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel1 single source.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Source: Input capture singal sourcce.
 *                This parameter can be a value of @ref CCT_ICSingalSourceEnum.
 *                  @arg CCT_IC_SINGAL_SOURCE_PIN: Used pin input as source.
 *                  @arg CCT_IC_SINGAL_SOURCE_ACMP: Used ACMP as source.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1SingleSourceConfig(CCT_Type* CCTx, CCT_ICSingalSourceEnum CCT_Source)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN1_SEL bit. */
  tmpReg &= ~CCT_CR_IN1_SEL_Msk;

  /* Set CCTx IN1_SEL bit according to source. */
  tmpReg |= (uint32_t)(CCT_Source << CCT_CR_IN1_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel1 ACMP channel.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Chl: Input capture analog channel.
 *                This parameter can be a value of @ref CCT_ICACMPChannelEnum.
 *                DPM32M08x|DPM32M05x|DPM32M03x:
 *                  @arg CCT_IC_ACMP0: Used ACMP0 as input capture source.
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 *                  @arg CCT_IC_ACMP2: Used ACMP2 as input capture source.
 *                  @arg CCT_IC_ACMP3: Used ACMP3 as input capture source.
 *                DPM32M036|DPM32M030:
 *                  @arg CCT_IC_ACMP0: Used ACMP0 as input capture source.
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 *                DPM32M015:
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1ACMPChannelConfig(CCT_Type* CCTx, CCT_ICACMPChannelEnum CCT_Chl)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN1_ACMP_SEL[1:0] bits. */
  tmpReg &= ~CCT_CR_IN1_ACMP_SEL_Msk;

  /* Set CCTx IN1_ACMP_SEL[1:0] bits according to chl. */
  tmpReg |= (uint32_t)(CCT_Chl << CCT_CR_IN1_ACMP_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel1 filter sample division.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Div: Input capture channel filter sample division.
 *                This parameter can be a value of @ref CCT_ICFiltSampleDivEnum.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV1: Filter sample 1 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV4: Filter sample 4 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV16: Filter sample 16 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV32: Filter sample 32 division.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1FltSampDivConfig(CCT_Type* CCTx, CCT_ICFiltSampleDivEnum CCT_Div)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH1_FLT_SAMPLE[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH1_FLT_SAMPLE_Msk;

  /* Set CCTx CH1_FLT_SAMPLE[1:0] bits according to div. */
  tmpReg |= (uint32_t)(CCT_Div << CCT_CAP_CFG_CH1_FLT_SAMPLE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel1 filter length.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Len: Input capture channel filter length.
 *                This parameter can be a value of @ref CCT_ICFiltLenEnum.
 *                  @arg CCT_IC_FILT_LEN1: Filter 1 length.
 *                  @arg CCT_IC_FILT_LEN8: Filter 8 length.
 *                  @arg CCT_IC_FILT_LEN16: Filter 16 length.
 *                  @arg CCT_IC_FILT_LEN32: Filter 32 length.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1FltLenConfig(CCT_Type* CCTx, CCT_ICFiltLenEnum CCT_Len)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH1_FLT_LEN[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH1_FLT_LEN_Msk;

  /* Set CCTx CH1_FLT_LEN[1:0] bits according to len. */
  tmpReg |= (uint32_t)(CCT_Len << CCT_CAP_CFG_CH1_FLT_LEN_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel1 action after capture.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Act: Input capture action after capture.
 *                This parameter can be a value of @ref CCT_ICActionEnum.
 *                  @arg CCT_IC_ACTION_NO: No action after capture.
 *                  @arg CCT_IC_ACTION_CLEAR_CNT: Clear counter register after capture.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1ActionConfig(CCT_Type* CCTx, CCT_ICActionEnum CCT_Act)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH1_CNT_CLR bit. */
  tmpReg &= ~CCT_CAP_CFG_CH1_CNT_CLR_Msk;

  /* Set CCTx CH1_CNT_CLR bit according to act. */
  tmpReg |= (uint32_t)(CCT_Act << CCT_CAP_CFG_CH1_CNT_CLR_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel1 capture polarity.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_Pol: Input capture polarity.
 *                This parameter can be a value of @ref CCT_ICPolarityEnum.
 *                  @arg CCT_IC_POLARITY_RISING: Capture rising edge.
 *                  @arg CCT_IC_POLARITY_FALLING: Capture falling edge.
 *                  @arg CCT_IC_POLARITY_RISINGFALLING: Capture rising and falling edge.
 * @retval  None.
 ******************************************************************************/
void CCT_IC1PolarityConfig(CCT_Type* CCTx, CCT_ICPolarityEnum CCT_Pol)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH1_EDGE[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH1_EDGE_Msk;

  /* Set CCTx CH1_EDGE[1:0] bits according to pol. */
  tmpReg |= (uint32_t)(CCT_Pol << CCT_CAP_CFG_CH1_EDGE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Get CH1 input capture value.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @retval  uint16_t: CH1 input capture value.
 ******************************************************************************/
uint16_t CCT_GetCapture1(CCT_Type* CCTx)
{
  return (uint16_t)CCTx->CH1_VALUE;
}

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 *******************************************************************************
 * @brief   Initializes the CCTx input capture channel2 according to the specified
 *          parameters in the CCT_ICInitType.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_ICInitType: Pointer to a CCT_ICInitTypeStruct structure that contains
 *                the configuration information for the specified CCT input capture.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2Init(CCT_Type* CCTx, CCT_ICInitTypeStruct* CCT_ICInitType)
{
  uint32_t tmpReg = 0, tmpReg1 = 0;
  /* Parameters check. */
  PARAM_ASSERT(IS_CCT_PERIPH(CCTx));
  PARAM_ASSERT(IS_CCT_IC_SINGLE_SOURCE(CCT_ICInitType->CCT_ICSingalSel));
  PARAM_ASSERT(IS_CCT_IC_ACMP_CHANNEL(CCT_ICInitType->CCT_ICACMPChl));
  PARAM_ASSERT(IS_CCT_IC_FILT_SAMPLE_DIV(CCT_ICInitType->CCT_ICFiltSampDiv));
  PARAM_ASSERT(IS_CCT_IC_FILT_LEN(CCT_ICInitType->CCT_ICFiltLen));
  PARAM_ASSERT(IS_CCT_IC_ACTION(CCT_ICInitType->CCT_ICAction));
  PARAM_ASSERT(IS_CCT_IC_POLARITY(CCT_ICInitType->CCT_ICPolarity));

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN2_ACMP_SEL[1:0],IN2_SEL,CH2_MODE bits. */
  tmpReg &= ~(CCT_CR_IN2_ACMP_SEL_Msk | CCT_CR_IN2_SEL_Msk | CCT_CR_CH2_MODE_Msk);

  /* Set CCTx IN2_ACMP_SEL[1:0] bits according to CCT_ICInitType. */
  tmpReg |= (uint32_t)(CCT_ICInitType->CCT_ICACMPChl << CCT_CR_IN2_ACMP_SEL_Pos);

  /* Set CCTx IN2_SEL bit according to CCT_ICInitType. */
  tmpReg |= (uint32_t)(CCT_ICInitType->CCT_ICSingalSel << CCT_CR_IN2_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;

  /* Read the value of the CCTx input capture control register. */
  tmpReg1 = CCTx->CAP_CFG;

  /* Clear CCTx CH2_FLT_SAMPLE[1:0],CH2_FLT_LEN[1:0],CH2_CNT_CLR,CH2_EDGE[1:0] bits. */
  tmpReg1 &= ~(CCT_CAP_CFG_CH2_FLT_SAMPLE_Msk | CCT_CAP_CFG_CH2_FLT_LEN_Msk \
               | CCT_CAP_CFG_CH2_CNT_CLR_Msk | CCT_CAP_CFG_CH2_EDGE_Msk);

  /* Set CCTx CH2_FLT_SAMPLE[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICFiltSampDiv << CCT_CAP_CFG_CH2_FLT_SAMPLE_Pos);

  /* Set CCTx CH2_FLT_LEN[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICFiltLen << CCT_CAP_CFG_CH2_FLT_LEN_Pos);

  /* Set CCTx CH2_CNT_CLR bit according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICAction << CCT_CAP_CFG_CH2_CNT_CLR_Pos);

  /* Set CCTx CH2_EDGE[1:0] bits according to CCT_ICInitType. */
  tmpReg1 |= (uint32_t)(CCT_ICInitType->CCT_ICPolarity << CCT_CAP_CFG_CH2_EDGE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg1;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel2 single source.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Source: Input capture singal sourcce.
 *                This parameter can be a value of @ref CCT_ICSingalSourceEnum.
 *                  @arg CCT_IC_SINGAL_SOURCE_PIN: Used pin input as source.
 *                  @arg CCT_IC_SINGAL_SOURCE_ACMP: Used ACMP as source.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2SingleSourceConfig(CCT_Type* CCTx, CCT_ICSingalSourceEnum CCT_Source)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN2_SEL bit. */
  tmpReg &= ~CCT_CR_IN2_SEL_Msk;

  /* Set CCTx IN2_SEL bit according to source. */
  tmpReg |= (uint32_t)(CCT_Source << CCT_CR_IN2_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel2 ACMP channel.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Chl: Input capture analog channel.
 *                This parameter can be a value of @ref CCT_ICACMPChannelEnum.
 *                  @arg CCT_IC_ACMP0: Used ACMP0 as input capture source.
 *                  @arg CCT_IC_ACMP1: Used ACMP1 as input capture source.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2ACMPChannelConfig(CCT_Type* CCTx, CCT_ICACMPChannelEnum CCT_Chl)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  /* Clear CCTx IN2_ACMP_SEL[1:0] bits. */
  tmpReg &= ~CCT_CR_IN2_ACMP_SEL_Msk;

  /* Set CCTx IN2_ACMP_SEL[1:0] bits according to chl. */
  tmpReg |= (uint32_t)(CCT_Chl << CCT_CR_IN2_ACMP_SEL_Pos);

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel2 filter sample division.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Div: Input capture channel filter sample division.
 *                This parameter can be a value of @ref CCT_ICFiltSampleDivEnum.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV1: Filter sample 1 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV4: Filter sample 4 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV16: Filter sample 16 division.
 *                  @arg CCT_IC_FILT_SAMPLE_DIV32: Filter sample 32 division.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2FltSampDivConfig(CCT_Type* CCTx, CCT_ICFiltSampleDivEnum CCT_Div)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH2_FLT_SAMPLE[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH2_FLT_SAMPLE_Msk;

  /* Set CCTx CH2_FLT_SAMPLE[1:0] bits according to div. */
  tmpReg |= (uint32_t)(CCT_Div << CCT_CAP_CFG_CH2_FLT_SAMPLE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel2 filter length.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Len: Input capture channel filter length.
 *                This parameter can be a value of @ref CCT_ICFiltLenEnum.
 *                  @arg CCT_IC_FILT_LEN1: Filter 1 length.
 *                  @arg CCT_IC_FILT_LEN8: Filter 8 length.
 *                  @arg CCT_IC_FILT_LEN16: Filter 16 length.
 *                  @arg CCT_IC_FILT_LEN32: Filter 32 length.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2FltLenConfig(CCT_Type* CCTx, CCT_ICFiltLenEnum CCT_Len)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH2_FLT_LEN[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH2_FLT_LEN_Msk;

  /* Set CCTx CH2_FLT_LEN[1:0] bits according to len. */
  tmpReg |= (uint32_t)(CCT_Len << CCT_CAP_CFG_CH2_FLT_LEN_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel2 action after capture.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Act: Input capture action after capture.
 *                This parameter can be a value of @ref CCT_ICActionEnum.
 *                  @arg CCT_IC_ACTION_NO: No action after capture.
 *                  @arg CCT_IC_ACTION_CLEAR_CNT: Clear counter register after capture.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2ActionConfig(CCT_Type* CCTx, CCT_ICActionEnum CCT_Act)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH2_CNT_CLR bit. */
  tmpReg &= ~CCT_CAP_CFG_CH2_CNT_CLR_Msk;

  /* Set CCTx CH2_CNT_CLR bit according to act. */
  tmpReg |= (uint32_t)(CCT_Act << CCT_CAP_CFG_CH2_CNT_CLR_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Configuration input capture channel2 capture polarity.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  CCT_Pol: Input capture polarity.
 *                This parameter can be a value of @ref CCT_ICPolarityEnum.
 *                  @arg CCT_IC_POLARITY_RISING: Capture rising edge.
 *                  @arg CCT_IC_POLARITY_FALLING: Capture falling edge.
 *                  @arg CCT_IC_POLARITY_RISINGFALLING: Capture rising and falling edge.
 * @retval  None.
 ******************************************************************************/
void CCT_IC2PolarityConfig(CCT_Type* CCTx, CCT_ICPolarityEnum CCT_Pol)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx input capture control register. */
  tmpReg = CCTx->CAP_CFG;

  /* Clear CCTx CH2_EDGE[1:0] bits. */
  tmpReg &= ~CCT_CAP_CFG_CH2_EDGE_Msk;

  /* Set CCTx CH2_EDGE[1:0] bits according to pol. */
  tmpReg |= (uint32_t)(CCT_Pol << CCT_CAP_CFG_CH2_EDGE_Pos);

  /* Store the new value. */
  CCTx->CAP_CFG = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Get CH2 input capture value.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @retval  uint16_t: CH1 input capture value.
 ******************************************************************************/
uint16_t CCT_GetCapture2(CCT_Type* CCTx)
{
  return (uint16_t)CCTx->CH2_VALUE;
}
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 *******************************************************************************
 * @brief   Enables or disables the CCTx capture/compare Channel0.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void CCT_CH0Cmd(CCT_Type* CCTx, FunctionalState NewState)
{
  if(DISABLE != NewState)
  {
    /* Enable CCTx CH0 output compare functions. */
    CCTx->CR |= CCT_CR_CH0_EN_Msk;
  }
  else
  {
    /* Disable CCTx CH0 output compare functions. */
    CCTx->CR &= ~CCT_CR_CH0_EN_Msk;
  }
}

/**
 *******************************************************************************
 * @brief   Enables or disables the CCTx capture/compare Channel1.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void CCT_CH1Cmd(CCT_Type* CCTx, FunctionalState NewState)
{
  if(DISABLE != NewState)
  {
    /* Enable CCTx CH1 output compare functions. */
    CCTx->CR |= CCT_CR_CH1_EN_Msk;
  }
  else
  {
    /* Disable CCTx CH1 output compare functions. */
    CCTx->CR &= ~CCT_CR_CH1_EN_Msk;
  }
}
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 *******************************************************************************
 * @brief   Enables or disables the CCTx capture/compare Channel2.
 * @param   [in]  CCTx: Where x can be 0 to select the CCT peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void CCT_CH2Cmd(CCT_Type* CCTx, FunctionalState NewState)
{
  if(DISABLE != NewState)
  {
    /* Enable CCTx CH2 output compare functions. */
    CCTx->CR |= CCT_CR_CH2_EN_Msk;
  }
  else
  {
    /* Disable CCTx CH2 output compare functions. */
    CCTx->CR &= ~CCT_CR_CH2_EN_Msk;
  }
}
#endif /* DPM32M036 */

#if defined (DPM32M08x) || defined (DPM32M05x)
/**
 *******************************************************************************
 * @brief   Enable or disable specifies DMA request.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_DMAReq: Specifies the DMA request.
 *                This parameter can be a value of @ref CCT_DMAReqEnum.
 *                  @arg CCT_DMA_REQ_OVERFLOW: Counter over flow DMA request.
 *                  @arg CCT_DMA_REQ_CH0_CAP: CH0 capture DMA request.
 *                  @arg CCT_DMA_REQ_CH1_CAP: CH1 capture DMA request.
 *                  @arg CCT_DMA_REQ_CH0_CMP: CH0 compare DMA request.
 *                  @arg CCT_DMA_REQ_CH1_CMP: CH1 compare DMA request.
 * @param   [in]  NewState: New state of the DMA request.
 *                This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void CCT_DMACmd(CCT_Type* CCTx, CCT_DMAReqEnum CCT_DMAReq, FunctionalState NewState)
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  if(DISABLE != NewState)
  {
    /* Set CCTx DMA_SRC_SEL[2:0] bits according to CCT_DMAReq. */
    tmpReg |= (uint32_t)(CCT_DMAReq << CCT_CR_DMA_SRC_SEL_Pos);
  }
  else
  {
    /* Clear CCTx DMA_SRC_SEL[2:0] bits. */
    tmpReg &= ~CCT_CR_DMA_SRC_SEL_Msk;
  }

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

#endif  /* DPM32M08x || DPM32M05x */

/**
 *******************************************************************************
 * @brief   Enable or disable specifies interrupt type.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_IntType: Specifies the intreeupt type.
 *                This parameter can be any combination of @ref CCT_IntTypeEnum.
 *                  @arg CCT_INT_TYPE_OVERFLOW: Counter overflow interrupt.
 *                  @arg CCT_INT_TYPE_CH0_CAP: CH0 capture interrupt.
 *                  @arg CCT_INT_TYPE_CH1_CAP: CH1 capture interrupt.
 *                  @arg CCT_INT_TYPE_CH2_CAP: CH2 capture interrupt.
 *                  @arg CCT_INT_TYPE_CH0_CMP: CH0 compare interrupt.
 *                  @arg CCT_INT_TYPE_CH1_CMP: CH1 compare interrupt.
 *                  @arg CCT_INT_TYPE_CH2_CMP: CH2 compare interrupt.
 * @param   [in]  NewState: New state of the interrupt.
 *                This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
void CCT_IntCmd(CCT_Type* CCTx, uint32_t CCT_IntType, FunctionalState NewState)
#elif defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
void CCT_IntCmd(CCT_Type* CCTx, uint16_t CCT_IntType, FunctionalState NewState)
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
{
  uint32_t tmpReg = 0UL;

  /* Read the value of the CCTx control register. */
  tmpReg = CCTx->CR;

  if(DISABLE != NewState)
  {
    /* Set CCTx INT bits according to CCT_IntType. */
    tmpReg |= (uint32_t)CCT_IntType;
  }
  else
  {
    /* Clear CCTx INT bits. */
    tmpReg &= ~(uint32_t)CCT_IntType;
  }

  /* Store the new value. */
  CCTx->CR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Get the specifies interrupt type enable or disable.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_IntType: Specifies the intreeupt type.
 *                This parameter can be a value of @ref CCT_IntTypeEnum.
 *                  @arg CCT_INT_TYPE_OVERLOW: Counter overflow interrupt.
 *                  @arg CCT_INT_TYPE_CH0_CAP: CH0 capture interrupt.
 *                  @arg CCT_INT_TYPE_CH1_CAP: CH1 capture interrupt.
 *                  @arg CCT_INT_TYPE_CH2_CAP: CH2 capture interrupt.
 *                  @arg CCT_INT_TYPE_CH0_CMP: CH0 compare interrupt.
 *                  @arg CCT_INT_TYPE_CH1_CMP: CH1 compare interrupt.
 *                  @arg CCT_INT_TYPE_CH2_CMP: CH2 compare interrupt.
 * @retval  FunctionalState: The CCT_IntType interrupt state(ENABLE or DISABLE).
 ******************************************************************************/
FunctionalState CCT_GetIntCmdStatus(CCT_Type* CCTx, CCT_IntTypeEnum CCT_IntType)
{
  FunctionalState state = DISABLE;

  if(RESET != (CCTx->CR & ((uint32_t)CCT_IntType)))
  {
    /* Interrupt is enable. */
    state = ENABLE;
  }
  else
  {
    /* Interrupt is disable. */
    state = DISABLE;
  }

  return state;
}

/**
 *******************************************************************************
 * @brief   Get the specified CCT flag is set or not.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_IntFlag: Specifies the CCT interrupt flag.
 *                This parameter can be a value of @ref CCT_IntFlagEnum.
 *                  @arg CCT_INT_FLAG_OVERFLOW: Counter over flow flag.
 *                  @arg CCT_INT_FLAG_CH0_CAPTURE: Channel0 capture flag.
 *                  @arg CCT_INT_FLAG_CH1_CAPTURE: Channel1 capture flag.
 *                  @arg CCT_INT_FLAG_CH2_CAPTURE: Channel2 capture flag.
 *                  @arg CCT_INT_FLAG_CH0_COMPARE: Channel0 compare flag.
 *                  @arg CCT_INT_FLAG_CH1_COMPARE: Channel1 compare flag.
 *                  @arg CCT_INT_FLAG_CH2_COMPARE: Channel2 compare flag.
 * @retval  FlagState: The new state of CCT_IntFlag (SET or RESET).
 ******************************************************************************/
FlagState CCT_GetIntFlagStatus(CCT_Type* CCTx, CCT_IntFlagEnum CCT_IntFlag)
{
  FlagState state = RESET;

  /* Get the status of the Interrupt */
  if(RESET != (CCTx->SR & ((uint32_t)CCT_IntFlag)))
  {
    state = SET;
  }
  else
  {
    state = RESET;
  }

  /* Return the status of the interrupt flag bit. */
  return state;
}

/**
 *******************************************************************************
 * @brief   Get the specified CCT flag is set or not.
 * @param   [in]  CCTx: Where x can be (0, 1) to select the CCT peripheral.
 * @param   [in]  CCT_IntFlag: Specifies the CCT interrupt flag.
 *                This parameter can be a value of @ref CCT_IntFlagEnum.
 *                  @arg CCT_INT_FLAG_OVERFLOW: Counter over flow flag.
 *                  @arg CCT_INT_FLAG_CH0_CAPTURE: Channel0 capture flag.
 *                  @arg CCT_INT_FLAG_CH1_CAPTURE: Channel1 capture flag.
 *                  @arg CCT_INT_FLAG_CH2_CAPTURE: Channel2 capture flag.
 *                  @arg CCT_INT_FLAG_CH0_COMPARE: Channel0 compare flag.
 *                  @arg CCT_INT_FLAG_CH1_COMPARE: Channel1 compare flag.
 *                  @arg CCT_INT_FLAG_CH2_COMPARE: Channel2 compare flag.
 * @retval  None.
 ******************************************************************************/
void CCT_ClearIntFlag(CCT_Type* CCTx, CCT_IntFlagEnum CCT_IntFlag)
{
  /* Clear interrupt flags.*/
  CCTx->SR = ((uint32_t)CCT_IntFlag);
}
