/**
 *******************************************************************************
 * @file    adc_scansample.c
 *
 * @brief   Implement system specific functions.
 *
 * @author  DP
 *
 * @version V1.0.1
 *
 * @date    2024-11-11
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "adc_scansample.h"

/**
  * @addtogroup DPM32M036_LibSamples
  * @{
  */

/**
  * @addtogroup ADC
  * @{
  */

/**
  * @addtogroup ADC ScanSample
  * @{
  */

/* Private typedef --------------------------------------------------------------*/
ADC_GPIOTypeStruct adcCHxGPIOMap[]  =
{
  {GPIOA, GPIO_PIN_0,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_1,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_2,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_3,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_4,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_5,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_6,GPIO_ANA_AF_0},
  {GPIOA, GPIO_PIN_7,GPIO_ANA_AF_0},
  {GPIOB, GPIO_PIN_0,GPIO_ANA_AF_0},
  {GPIOB, GPIO_PIN_1,GPIO_ANA_AF_0},
};

uint8_t adcChannelMap[] = {ADC_Channel_0, ADC_Channel_1, ADC_Channel_2, ADC_Channel_3,
                           ADC_Channel_4, ADC_Channel_5, ADC_Channel_6, ADC_Channel_7,
                           ADC_Channel_8, ADC_Channel_9};

GPIO_InitTypeStruct ADC_GPIO_InitStruct;
ADCCOM_InitTypeStruct ADCCOM_InitStruct;
ADCCOM_ChannelStruct ADCCOM_ChannelConfigStruct;
ADC_InitTypeStruct ADC_InitStruct;
ADC_ChannelStruct ADC_ChannelConfigStruct;

/* Private define ---------------------------------------------------------------*/
#define ChanleNum 10

/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/
volatile uint16_t ADC_ScanCode[ChanleNum] = {0};

/* Private functions ------------------------------------------------------------*/
/**
 *******************************************************************************
 * @brief   ADC0 IRQ Handler.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void ADC_IRQ_Handler(void)
{
  if(ADC_GetIntCmdStatus(ADC0, ADC_INT_TYPE_EOS) == SET && ADC_GetIntFlagStatus(ADC0, ADC_INT_FLAG_EOS) == SET)
  {
    /* Clear flag */
    ADC_ClearIntFlag(ADC0, ADC_INT_FLAG_EOS);
    
    for(uint8_t i=0; i<ChanleNum; i++)
    {
      ADC_ScanCode[i] = ADC_GetScanData(ADC0,i);
    }
  }
}

/**
 *******************************************************************************
 * @brief   ADC GPIO Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void ADC_GPIO_Init(ADC_GPIOTypeStruct *ADC_GPIOStruct)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE);
  RCC_Lock();
  GPIO_StructInit(&ADC_GPIO_InitStruct);
  ADC_GPIO_InitStruct.GPIO_Mode = GPIO_MODE_ANALOG;
  ADC_GPIO_InitStruct.GPIO_Speed = GPIO_SPEED_HIGH;
  ADC_GPIO_InitStruct.GPIO_OType = GPIO_OTYPE_PP;
  ADC_GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_NOPULL;
  ADC_GPIO_InitStruct.GPIO_Pin = ADC_GPIOStruct->pin;
  GPIO_Init(ADC_GPIOStruct->gpio, &ADC_GPIO_InitStruct);
  GPIO_AnaAFConfig(ADC_GPIOStruct->gpio, ADC_GPIOStruct->pin, ADC_GPIOStruct->ana_func);
}
/**
 *******************************************************************************
 * @brief   ADC Config.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void ADC_Config(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_ADC, ENABLE);
  RCC_Lock();
  
  /* Config ADC common parameters */
  ADCCOM_StructInit(&ADCCOM_InitStruct);
  ADCCOM_InitStruct.AdcSpeed = 1;
  ADCCOM_InitStruct.Vref = ADC_VREF_AVDD;
  ADCCOM_InitStruct.ClkDiv = ADC_CLK_DIV1;
  ADCCOM_InitStruct.SampleTime0 = ADC_SAMPLE_TIME_64Cycles;
  ADCCOM_InitStruct.SampleTime1 = ADC_SAMPLE_TIME_64Cycles;
  ADCCOM_InitStruct.SampleAccuracy = ADC_Sample_Accuracy_12Bit;
  ADCCOM_InitStruct.IN15Sel = ADC_IN15_SEL_TEMP;
  ADCCOM_Init(&ADCCOM_InitStruct);
  ADCCOM_ChannelStructInit(&ADCCOM_ChannelConfigStruct);

  /* Config all channel */
  for(uint8_t i = 0; i < ChanleNum; i++)
  {
    ADCCOM_ChannelConfigStruct.Channel = adcChannelMap[i];
    ADCCOM_ChannelConfigStruct.OversampleEnable = DISABLE;
    ADCCOM_ChannelConfigStruct.SampleTimeSel = ADC_SAMPLE_TIME_SEL0;
    ADCCOM_ChannelConfigStruct.Offset = 0;
    ADCCOM_ChannelConfig(&ADCCOM_ChannelConfigStruct);
    ADC_ChannelStructInit(&ADC_ChannelConfigStruct);
    ADC_ChannelConfigStruct.Channel = adcChannelMap[i];
    ADC_ChannelConfigStruct.DataAlign = ADC_DATA_ALIGN_RIGHT;
    ADC_ChannelConfigStruct.DataSign = ADC_DATA_UNSIGNED;
    ADC_ChannelConfig(ADC0, &ADC_ChannelConfigStruct);
  }

  /* Config ADC0 */
  ADC_StructInit(&ADC_InitStruct);
  /* Single sequence scan */
  ADC_InitStruct.TrigMode = ADC_TRIG_HW_SEQ_CONV;
  ADC_InitStruct.ContinueWait = ADC_CONTINUE_WAIT_CYCLE_16;
  /* HW0_SRC is used to trigger a single sequence scan */
  ADC_InitStruct.HardwareTrig0.TrigSource = ADC_TRIG_SOURCE_TIMER0;
  ADC_InitStruct.HardwareTrig0.TrigEnable = ENABLE;
  ADC_InitStruct.ScanChannelLen = ChanleNum;
  ADC_Init(ADC0, &ADC_InitStruct);
  
  ADC_IntCmd(ADC0,ADC_INT_TYPE_EOS,ENABLE);
  NVIC_EnableIRQ(ADC0_IRQn);  
}
/**
 *******************************************************************************
 * @brief   ADC hardware tirgger Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void ADC_HardwareTrigger_Init(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_TIM0, ENABLE);
  RCC_Lock();
  
  /* Config TIM0 is used to trigger a single scan */
  TIM_InitTypeStruct TIM_InitStruct;
  TIM_StructInit(&TIM_InitStruct);
  TIM_InitStruct.TIM_ReloadValue = 48000 * 1000; 
  TIM_InitStruct.TIM_CounterCycle = TIM_COUNTER_CYCLE_PERIOD;
  TIM_InitStruct.TIM_IntState = DISABLE;
  TIM_Init(TIM0, &TIM_InitStruct);
  TIM_Cmd(TIM0, ENABLE);
}

/**
 *******************************************************************************
 * @brief   ADC ScanSample Sample.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void ADC_ScanSample_Sample(void)
{
  /* ADC config */
  ADC_Config();

  /* Config ADC channel GPIO */
  for(uint8_t i = 0; i < ChanleNum; i++)
  {
    ADC_GPIO_Init(&adcCHxGPIOMap[i]);

    ADC_ScanChannelConfig(ADC0, i, i);
  }
  
  /* ADC hardware tirgger Init */  
  ADC_HardwareTrigger_Init();

  /* Enable ADC0 */
  ADC_Cmd(ADC0, ENABLE);

  /* Start conversation */
  ADC_StartConversion(ADC0);

}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
