/**
 *******************************************************************************
 * @file    uart_interrupt.c
 *
 * @brief   Implement system specific functions.
 *
 * @author  DP
 *
 * @version V1.0.1
 *
 * @date    2024-11-11
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "uart_interrupt.h"

/**
  * @addtogroup DPM32M036_LibSamples
  * @{
  */

/**
  * @addtogroup UART
  * @{
  */

/**
  * @addtogroup UART Interrupt
  * @{
  */

/* Private typedef --------------------------------------------------------------*/

/* Private define ---------------------------------------------------------------*/
#define ARRAY_NUM(arr_nanme)      (uint32_t)(sizeof(arr_nanme) / sizeof(*(arr_nanme)))
#define TX_SIZE                   (ARRAY_NUM(tx_buf) - 1)
/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/
uint8_t tx_buf[] = "Hello DPM32. Received, please reply\r\n";
uint8_t rx_buf[30] = {0};

uint8_t tx_size = TX_SIZE;
uint8_t rx_size = 30;
uint8_t tx_count = 0; 
uint8_t rx_count = 0;
/* Private functions ------------------------------------------------------------*/
/**
 *******************************************************************************
 * @brief   UART0 IRQ Handler.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void UART0_IRQ_Handler(void)
{
  if(UART_GetIntCmdStatus(UART0, UART_INT_RXNE_IE) != DISABLE && UART_GetIntFlagStatus(UART0, UART_INT_FLAG_RXNE) != RESET)
  {
    UART_ClearIntFlag(UART0, UART_INT_FLAG_RXNE);
    /* receive data */
    if((rx_count < rx_size))
    {
      rx_buf[rx_count] = UART_ReceiveData(UART0);
      /* printf receive data */
      UART_SendData(UART0,rx_buf[rx_count]);
    }
    rx_count++;
  }

  if(UART_GetIntCmdStatus(UART0, UART_INT_TXE_IE) != DISABLE && UART_GetIntFlagStatus(UART0, UART_INT_FLAG_TXE) != RESET)
  {
    UART_ClearIntFlag(UART0, UART_INT_FLAG_TXE);
    /* transmit data */
    if((tx_count < tx_size))
    {
      UART_SendData(UART0,tx_buf[tx_count]);
    }
    tx_count++;
  }
}

/**
 *******************************************************************************
 * @brief   UART GPIO Config.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void UART_GPIOConfig(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE);
  RCC_Lock();

  /* connect port to UARTx Pin */
  GPIO_InitTypeStruct GPIO_InitStruct;
  GPIO_StructInit(&GPIO_InitStruct);
  GPIO_InitStruct.GPIO_Mode = GPIO_MODE_ALT_FUNC;
  GPIO_InitStruct.GPIO_Speed = GPIO_SPEED_HIGH;
  GPIO_InitStruct.GPIO_OType =  GPIO_OTYPE_PP;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_NOPULL;

  GPIO_InitStruct.GPIO_Pin =  GPIO_PIN_9 | GPIO_PIN_10;
  GPIO_Init(GPIOA, &GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOA, GPIO_PIN_9 | GPIO_PIN_10, GPIO_ALT_FUNC_1);
}

/**
 *******************************************************************************
 * @brief   UART Config.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void UART_ConfigInit(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_UART0, ENABLE);
  RCC_Lock();

  /* UART configure */
  UART_DeInit(UART0);
  UART_InitTypeStruct UART_InitStruct;
  UART_StructInit(&UART_InitStruct);
  UART_InitStruct.UART_BaudRate = 115200;
  UART_InitStruct.UART_DataLen = UART_DATA_LEN_8;
  UART_InitStruct.UART_StopBits = UART_STOP_BITS_1;
  UART_InitStruct.UART_Parity = UART_PARITY_NO;
  UART_InitStruct.UART_Mode = UART_MODE_RX_TX;
  UART_Init(UART0, &UART_InitStruct);
  UART_Cmd(UART0, ENABLE);
  
  /* Enable UART0 interrupt */
  UART_IntCmd(UART0, UART_INT_RXNE_IE, ENABLE);
  UART_IntCmd(UART0, UART_INT_TXE_IE, ENABLE);
  NVIC_SetPriority(UART0_IRQn, 2);
  NVIC_EnableIRQ(UART0_IRQn);

}
/**
 *******************************************************************************
 * @brief   UART Interrupt Sample.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void UART_Interrupt_Sample(void)
{
  /* Config uart0 GPIO */ 
  UART_GPIOConfig();
  
  /* Config uart0 */
  UART_ConfigInit();
}

/* retarget the C library printf function to the usart */
int fputc(int ch, FILE *f)
{
  UART_SendData(UART0, (uint8_t)ch);

  while(RESET == UART_GetIntFlagStatus(UART0, UART_INT_FLAG_TXE));

  return ch;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
