/**
 *******************************************************************************
 * @file    dpm32m0xx_math_acc.h
 *
 * @brief   Header file for MATH_ACC firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_MATH_ACC_H__
#define __DPM32M0XX_MATH_ACC_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"

#if defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036)

#define MATH_ACC_TIMEOUT_VALUE     0x0040      /* Calculate timeout value of MATH_ACC. */

/**
 ******************************************************************************
 * @brief MATH_ACC mode enumeration.
 *****************************************************************************/
typedef enum
{
#if defined (DPM32M036)
  MATH_ACC_MODE_TRIANGLE          = 0x0UL,      /* Calculate triangle functions. */
  MATH_ACC_MODE_ARCTANGENT_MOD    = 0x1UL,      /* Calculating the arctangent and taking the modulus. */
  MATH_ACC_MODE_DIVISION          = 0x2UL       /* Calculate division. */
#else /* DPM32M05x || DPM32M03x */
  MATH_ACC_MODE_TRIANGLE          = 0x8FEEUL,   /* Calculate triangle functions. */
  MATH_ACC_MODE_ARCTANGENT_MOD    = 0x9FFEUL,   /* Calculating the arctangent and taking the modulus. */
  MATH_ACC_MODE_DIVISION          = 0xAEEFUL    /* Calculate division. */
#endif /* DPM32M036 */
} MATH_ACC_ModeEnum;

/**
 ******************************************************************************
 * @brief MATH_ACC calculate status enumeration.
 *****************************************************************************/
typedef enum
{
  MATH_ACC_FLAG_BUSY         = MATH_ACC_SR_BUSY_Msk,          /* Calculate busy. */
  MATH_ACC_FLAG_DIV_ZERO     = MATH_ACC_SR_DIV_ZERO_IF_Msk    /* Calculate division zero. */
} MATH_ACC_FlagEnum;

/**
 ******************************************************************************
 * @brief MATH_ACC calculate result enumeration.
 *****************************************************************************/
typedef enum
{
  MATH_ACC_CAL_RESULT_FAIL       = 0,                           /*!< Calculate fail. */
  MATH_ACC_CAL_RESULT_SUCCESS    = !MATH_ACC_CAL_RESULT_FAIL    /*!< Calculate success. */
} MATH_ACC_CalResultEnum;

/* Deinitializes and initialization functions. */
void                    MATH_ACC_Cmd(FunctionalState NewState);

/* Calculate process management functions. */
void                    MATH_ACC_SetMode(MATH_ACC_ModeEnum MATH_ACC_Mode);

#if defined (DPM32M036)
void                    MATH_ACC_SetOP0Data(int32_t MATH_ACC_OP0);
void                    MATH_ACC_SetOP1Data(int32_t MATH_ACC_OP1);
int32_t                 MATH_ACC_GetOP0Data(void);
int32_t                 MATH_ACC_GetOP1Data(void);
#else
void                    MATH_ACC_SetOP0Data(int16_t MATH_ACC_OP0);
void                    MATH_ACC_SetOP1Data(int16_t MATH_ACC_OP1);
int16_t                 MATH_ACC_GetOP0Data(void);
int16_t                 MATH_ACC_GetOP1Data(void);
#endif /* DPM32M036 */
void                    MATH_ACC_StartCalculate(void);
FlagState               MATH_ACC_GetFlagStatus(MATH_ACC_FlagEnum MATH_ACC_Flag);
void                    MATH_ACC_ClearDivZeroFlag(void);


/* Calculate parameter management functions. */
MATH_ACC_CalResultEnum  MATH_ACC_CalTriangle(int16_t MATH_ACC_Anagle, int16_t* MATH_ACC_Sin, int16_t* MATH_ACC_Cos);
MATH_ACC_CalResultEnum  MATH_ACC_CalArctAndMod(int16_t MATH_ACC_Sin, int16_t MATH_ACC_Cos, \
    int16_t* MATH_ACC_Arct, int16_t* MATH_ACC_Mod);
#if defined (DPM32M036)
MATH_ACC_CalResultEnum  MATH_ACC_CalDivision(int32_t MATH_ACC_Divisor, int32_t MATH_ACC_Dividend, \
    int32_t* MATH_ACC_Quotations, int32_t* MATH_ACC_Remainder);
#else
MATH_ACC_CalResultEnum  MATH_ACC_CalDivision(int16_t MATH_ACC_Divisor, int16_t MATH_ACC_Dividend, \
                                             int16_t* MATH_ACC_Quotations, int16_t* MATH_ACC_Remainder);
#endif /* DPM32M036 */

/* MATH_ACC interrupt and flags management functions. */
void                    MATH_ACC_IntCmd(FunctionalState NewState);
FunctionalState         MATH_ACC_GetIntCmdStatus(void);
FlagState               MATH_ACC_GetIntFlagStatus(void);
void                    MATH_ACC_ClearIntFlag(void);

#if defined (DPM32M036)
void                    MATH_ACC_DivZeroIntCmd(FunctionalState NewState);
FunctionalState         MATH_ACC_GetDivZeroIntCmdStatus(void);
FlagState               MATH_ACC_GetDivZeroIntFlagStatus(void);
void                    MATH_ACC_ClearDivZeroIntFlag(void);
#endif /* DPM32M036 */



#endif /* DPM32M05x || DPM32M03x || DPM32M036 */

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif


#endif /* __DPM32M0XX_MATH_ACC_H__ */
