/**
 *******************************************************************************
 * @file    dpm32m0xx_acmp.h
 *
 * @brief   Header file for ACMP firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_ACMP_H__
#define __DPM32M0XX_ACMP_H__

#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif

#include "dpm32m0xx.h"


#define ACMP_CHCFG(channel)             REG32((ACMP_BASE + 0x08UL) + 0x04UL*(channel))   /*!< the address of ACMP channel configuration register */

/**
 ******************************************************************************
 * @brief ACMP source of reference voltage enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_VREF_VDD         = 0x00UL,                   /*!< Selecting VDD as the source of reference voltage. */
  ACMP_VREF_BANDGAP     = ACMP_CR_VREF_SEL_Msk      /*!< Select BandGap as the source of reference voltage. */
} ACMP_VrefSelEnum;

#if defined (DPM32M036) || defined (DPM32M030)
/**
 ******************************************************************************
 * @brief ACMP BEMFR of input 0 select enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_BEMFR_INPUT0_ACMP_IN0         = 0x000UL,      /*!< Select ACMP_IN0 as the input 0 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT0_ACMP_IN1         = 0x100UL,      /*!< Select ACMP_IN1 as the input 0 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT0_ACMP_IN2         = 0x200UL,      /*!< Select ACMP_IN2 as the input 0 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT0_ACMP_IN3         = 0x300UL       /*!< Select ACMP_IN3 as the input 0 of the back BEMFR resistance circuit. */
} ACMP_BEMFRInput0Enum;

/**
 ******************************************************************************
 * @brief ACMP BEMFR of input 1 select enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_BEMFR_INPUT1_ACMP_IN1         = 0x000UL,      /*!< Select ACMP_IN1 as the input 1 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT1_ACMP_IN2         = 0x400UL,      /*!< Select ACMP_IN2 as the input 1 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT1_ACMP_IN3         = 0x800UL,      /*!< Select ACMP_IN3 as the input 1 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT1_ACMP_IN4         = 0xC00UL       /*!< Select ACMP_IN4 as the input 1 of the back BEMFR resistance circuit. */
} ACMP_BEMFRInput1Enum;

/**
 ******************************************************************************
 * @brief ACMP BEMFR of input 2 select enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_BEMFR_INPUT2_ACMP_IN2         = 0x0000UL,     /*!< Select ACMP_IN2 as the input 2 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT2_ACMP_IN3         = 0x1000UL,     /*!< Select ACMP_IN3 as the input 2 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT2_ACMP_IN4         = 0x2000UL,     /*!< Select ACMP_IN4 as the input 2 of the back BEMFR resistance circuit. */
  ACMP_BEMFR_INPUT2_ACMP_IN5         = 0x3000UL      /*!< Select ACMP_IN5 as the input 2 of the back BEMFR resistance circuit. */
} ACMP_BEMFRInput2Enum;
#endif /* DPM32M036 || DPM32M030 */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
/**
 ******************************************************************************
 * @brief ACMP overcurrent protection input select enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_OCP_INPUT_PGA0_OUT       = 0x00000UL,         /*!< Select PGA0_OUT as overcurrent protection input. */
#if defined (DPM32M036) || defined (DPM32M030)
  ACMP_OCP_INPUT_PGA1_OUT       = 0x10000UL,         /*!< Select PGA1_OUT as overcurrent protection input. */
#endif /* DPM32M036 || DPM32M030 */
  ACMP_OCP_INPUT_PGA0_IN_P      = 0x20000UL,         /*!< Select PGA0_IN_P as overcurrent protection input. */
  ACMP_OCP_INPUT_PGA0_IN_N      = 0x30000UL,         /*!< Select PGA0_IN_N as overcurrent protection input. */
#if defined (DPM32M036) || defined (DPM32M030)
  ACMP_OCP_INPUT_PGA1_IN_P      = 0x40000UL,         /*!< Select PGA1_IN_P as overcurrent protection input. */
  ACMP_OCP_INPUT_PGA1_IN_N      = 0x50000UL,         /*!< Select PGA1_IN_N as overcurrent protection input. */
#endif /* DPM32M036 || DPM32M030 */
#if defined (DPM32M036) 
  ACMP_OCP_INPUT_PGA2_IN_P      = 0x60000UL,         /*!< Select PGA2_IN_P as overcurrent protection input. */
  ACMP_OCP_INPUT_PGA2_IN_N      = 0x70000UL,         /*!< Select PGA2_IN_N as overcurrent protection input. */
#endif /* DPM32M036 */
} ACMP_OCPInputEnum;
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 ******************************************************************************
 * @brief ACMP Resistance select enumeration.Select the voltage divider value
          of the reference voltage Vref as the negative input of the comparator.
 *****************************************************************************/
typedef enum
{
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  ACMP_RES_SEL1         = 0x00UL,                    /*!< Select 1/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL2         = 0x01UL,                    /*!< Select 2/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL3         = 0x02UL,                    /*!< Select 3/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL4         = 0x03UL,                    /*!< Select 4/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL5         = 0x04UL,                    /*!< Select 5/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL6         = 0x05UL,                    /*!< Select 6/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL7         = 0x06UL,                    /*!< Select 7/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL8         = 0x07UL,                    /*!< Select 8/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL9         = 0x08UL,                    /*!< Select 9/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL10        = 0x09UL,                    /*!< Select 10/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL11        = 0x0AUL,                    /*!< Select 11/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL12        = 0x0BUL,                    /*!< Select 12/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL13        = 0x0CUL,                    /*!< Select 13/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL14        = 0x0DUL,                    /*!< Select 14/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL15        = 0x0EUL,                    /*!< Select 15/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL20        = 0x0FUL,                    /*!< Select 20/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL25        = 0x10UL,                    /*!< Select 25/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL30        = 0x11UL,                    /*!< Select 30/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL35        = 0x12UL,                    /*!< Select 35/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL40        = 0x13UL,                    /*!< Select 40/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL45        = 0x14UL,                    /*!< Select 45/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL50        = 0x15UL,                    /*!< Select 50/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL55        = 0x16UL,                    /*!< Select 55/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL60        = 0x17UL,                    /*!< Select 60/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL65        = 0x18UL,                    /*!< Select 65/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL70        = 0x19UL,                    /*!< Select 70/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL75        = 0x1AUL,                    /*!< Select 75/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL80        = 0x1BUL,                    /*!< Select 80/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL85        = 0x1CUL,                    /*!< Select 85/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL90        = 0x1DUL,                    /*!< Select 90/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL95        = 0x1EUL,                    /*!< Select 95/100 of the VDD voltage as the negative input of the comparator. */
  ACMP_RES_SEL100       = 0x1FUL,                    /*!< Select 100/100 of the VDD voltage as the negative input of the comparator. */
#else
  ACMP_RES_SEL2         = 0x00UL,                    /*!< Select 2/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL3         = 0x01UL,                    /*!< Select 3/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL4         = 0x02UL,                    /*!< Select 4/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL5         = 0x03UL,                    /*!< Select 5/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL6         = 0x04UL,                    /*!< Select 6/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL7         = 0x05UL,                    /*!< Select 7/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL8         = 0x06UL,                    /*!< Select 8/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL9         = 0x07UL,                    /*!< Select 9/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL10        = 0x08UL,                    /*!< Select 10/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL11        = 0x09UL,                    /*!< Select 11/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL12        = 0x0AUL,                    /*!< Select 12/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL13        = 0x0BUL,                    /*!< Select 13/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL14        = 0x0CUL,                    /*!< Select 14/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL15        = 0x0DUL,                    /*!< Select 15/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL16        = 0x0EUL,                    /*!< Select 16/20 of the reference voltage as the negative input of the comparator. */
  ACMP_RES_SEL17        = ACMP_CR_RES_SEL_Msk        /*!< Select 17/20 of the reference voltage as the negative input of the comparator. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
} ACMP_ResSelEnum;


/**
 ******************************************************************************
 * @brief ACMP channel enumeration.
 *****************************************************************************/
typedef enum
{
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036) || defined (DPM32M030) 
  ACMP_CH0         = 0x00UL,   /*!< ACMP channel0 . */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  ACMP_CH1         = 0x01UL,   /*!< ACMP channel1 . */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  ACMP_CH2         = 0x02UL,   /*!< ACMP channel2 . */
  ACMP_CH3         = 0x03UL    /*!< ACMP channel3 . */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
} ACMP_ChannelEnum;


/**
 ******************************************************************************
 * @brief ACMP output polarity enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_POL_NORMAL      = 0x00UL,                   /*!< Comparator results normal output. */
#if defined (DPM32M015) 
  ACMP_POL_INVERT      = ACMP_ACMP1_CFG_POL_Msk    /*!< Comparator results invert output. */
#else /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  ACMP_POL_INVERT      = ACMP_ACMP0_CFG_POL_Msk    /*!< Comparator results invert output. */
#endif /* DPM32M015 */
} ACMP_PolarityEnum;

/**
 ******************************************************************************
 * @brief ACMP interrupt trigger type enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_INT_TYPE_RISING             = 0x00UL,             /*!< The comparison result output rising edge trigger interrupt. */
  ACMP_INT_TYPE_FALLING            = 0x10UL,             /*!< The comparison result output falling edge trigger interrupt. */
  ACMP_INT_TYPE_RISING_FALLING     = 0x20UL              /*!< The comparison result output rising and falling edge trigger interrupt. */
} ACMP_IntTypeEnum;

/**
 ******************************************************************************
 * @brief ACMP Filter Sample Frequency enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_FILTER_SAMPLE_DIV1    = 0x0000UL,                       /*!< The ACMP filter sampling clock is equal to PCLK. */
  ACMP_FILTER_SAMPLE_DIV4    = 0x0100UL,                       /*!< The ACMP filter sampling clock is equal to PCLK/4. */
  ACMP_FILTER_SAMPLE_DIV8    = 0x0200UL,                       /*!< The ACMP filter sampling clock is equal to PCLK/8. */
#if defined (DPM32M015) 
  ACMP_FILTER_SAMPLE_DIV16   = ACMP_ACMP1_CFG_FLT_SAMPLE_Msk   /*!< The ACMP filter sampling clock is equal to PCLK/16. */
#else /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  ACMP_FILTER_SAMPLE_DIV16   = ACMP_ACMP0_CFG_FLT_SAMPLE_Msk   /*!< The ACMP filter sampling clock is equal to PCLK/16. */
#endif /* DPM32M015 */
} ACMP_FilterSampleEnum;

/**
 ******************************************************************************
 * @brief ACMP Filter length enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_FILTER_LEN1         = 0x0000UL,                     /*!< Filter samples 1 cycles length. */
  ACMP_FILTER_LEN8         = 0x0400UL,                     /*!< Filter samples 8 cycles length. */
  ACMP_FILTER_LEN16        = 0x0800UL,                     /*!< Filter samples 16 cycles length. */
#if defined (DPM32M015) 
  ACMP_FILTER_LEN32        = ACMP_ACMP1_CFG_FLT_LEN_Msk    /*!< Filter samples 32 cycles length. */
#else /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  ACMP_FILTER_LEN32        = ACMP_ACMP0_CFG_FLT_LEN_Msk    /*!< Filter samples 32 cycles length. */
#endif /* DPM32M015 */
} ACMP_FilterLenEnum;


/**
 ******************************************************************************
 * @brief ACMP positive input sourec enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_IN_P_SEL_ACMP_IN0    = 0x000000UL,                    /*!< ACMP positive input select ACMP_IN0. */
  ACMP_IN_P_SEL_ACMP_IN1    = 0x010000UL,                    /*!< ACMP positive input select ACMP_IN1. */
  ACMP_IN_P_SEL_ACMP_IN2    = 0x020000UL,                    /*!< ACMP positive input select ACMP_IN2. */
  ACMP_IN_P_SEL_ACMP_IN3    = 0x030000UL,                    /*!< ACMP positive input select ACMP_IN3. */
  ACMP_IN_P_SEL_ACMP_IN4    = 0x040000UL,                    /*!< ACMP positive input select ACMP_IN4. */
  ACMP_IN_P_SEL_ACMP_IN5    = 0x050000UL,                    /*!< ACMP positive input select ACMP_IN5. */
  ACMP_IN_P_SEL_PGA0        = 0x060000UL,                    /*!< ACMP positive input select PGA0.     */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036) || defined (DPM32M030)
  ACMP_IN_P_SEL_PGA1        = 0x070000UL,                    /*!< ACMP positive input select PGA1.     */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  /* When selecting ACMP1, the positive output can choose PGA2_OUT and OCP_IN, without PGA0_OUT and PGA1_OUT options  */
#if defined (DPM32M036) 
  ACMP_IN_P_SEL_PGA2        = 0x060000UL,                    /*!< ACMP1 positive input select PGA2.    */
#endif /* DPM32M036 */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  ACMP_IN_P_SEL_OCP         = 0x070000UL                     /*!< ACMP1 positive input select OCP.     */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
} ACMP_InputPSelEnum;

/**
 ******************************************************************************
 * @brief ACMP negative input sourec enumeration.
 *****************************************************************************/
typedef enum
{
  ACMP_IN_N_SEL_ACMP_IN0         = 0x000000UL,                    /*!< ACMP negative input select ACMP_IN0. */
  ACMP_IN_N_SEL_ACMP_IN1         = 0x100000UL,                    /*!< ACMP negative input select ACMP_IN1. */
  ACMP_IN_N_SEL_ACMP_IN2         = 0x200000UL,                    /*!< ACMP negative input select ACMP_IN2. */
  ACMP_IN_N_SEL_ACMP_IN3         = 0x300000UL,                    /*!< ACMP negative input select ACMP_IN3. */
  ACMP_IN_N_SEL_ACMP_IN4         = 0x400000UL,                    /*!< ACMP negative input select ACMP_IN4. */
  ACMP_IN_N_SEL_ACMP_IN5         = 0x500000UL,                    /*!< ACMP negative input select ACMP_IN5. */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  ACMP_IN_N_SEL_DAC_OUT          = 0x600000UL,                    /*!< ACMP negative input select DAC_OUT . */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
#if defined (DPM32M015) 
  ACMP_IN_N_SEL_VREF_DIV         = ACMP_ACMP1_CFG_IN_N_SEL_Msk    /*!< ACMP negative input select VREF_DIV. */
#else /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */
  ACMP_IN_N_SEL_VREF_DIV         = ACMP_ACMP0_CFG_IN_N_SEL_Msk    /*!< ACMP negative input select VREF_DIV. */
#endif /* DPM32M015 */
} ACMP_InputNSelEnum;

/**
 ******************************************************************************
 * @brief ACMP filter initialization structure definition.
 *****************************************************************************/
typedef struct
{
  ACMP_FilterSampleEnum          FilterSample;   /*!< ACMP filter sample frequency. */
  ACMP_FilterLenEnum             FilterLen;      /*!< ACMP filter length. */
  FunctionalState                FilterCmd;      /*!< ACMP filter enable. */
} ACMP_FilterInitTypeStruct;

/**
 ******************************************************************************
 * @brief ACMP initialization structure definition.
 *****************************************************************************/
typedef struct
{
  ACMP_PolarityEnum              ACMP_Polarity;           /*!< ACMP polarity select. */
  ACMP_InputPSelEnum             ACMP_PInput;             /*!< ACMP Positive input select. */
  ACMP_InputNSelEnum             ACMP_NInput;             /*!< ACMP Negative input select. */
  ACMP_FilterInitTypeStruct      ACMP_Filter;             /*!< ACMP filter configuration. */
  FunctionalState                ACMP_HYSCmd;             /*!< ACMP HYS enable. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
  FunctionalState                ACMP_HoldCmd;            /*!< ACMP result hold enable. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
} ACMP_InitTypeStruct;

/* ACMP Param Check. */
#define IS_ACMP_VREF(x) (((x) == ACMP_VREF_VDD) || \
                         ((x) == ACMP_VREF_BANDGAP))
                         
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define IS_ACMP_CHANNEL(x) (((x) == ACMP_CH0) || \
                            ((x) == ACMP_CH1)  || \
                            ((x) == ACMP_CH2)  || \
                            ((x) == ACMP_CH3))
#elif defined (DPM32M036) || defined (DPM32M030)
#define IS_ACMP_CHANNEL(x) (((x) == ACMP_CH0) || \
                            ((x) == ACMP_CH1))
#elif defined (DPM32M015)
#define IS_ACMP_CHANNEL(x) (((x) == ACMP_CH1))
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#define IS_ACMP_POLARITY(x) (((x) == ACMP_POL_NORMAL)    || \
                             ((x) == ACMP_POL_INVERT))

#define IS_ACMP_INT_TYPE(x) (((x) == ACMP_INT_TYPE_RISING)    || \
                             ((x) == ACMP_INT_TYPE_FALLING)  || \
                             ((x) == ACMP_INT_TYPE_RISING_FALLING))

#define IS_ACMP_FILTER_SAMPLE(x) (((x) == ACMP_FILTER_SAMPLE_DIV1)  || \
                                  ((x) == ACMP_FILTER_SAMPLE_DIV4)  || \
                                  ((x) == ACMP_FILTER_SAMPLE_DIV8)  || \
                                  ((x) == ACMP_FILTER_SAMPLE_DIV16))

#define IS_ACMP_FILTER_LEN(x) (((x) == ACMP_FILTER_LEN1)   || \
                               ((x) == ACMP_FILTER_LEN8)   || \
                               ((x) == ACMP_FILTER_LEN16)  || \
                               ((x) == ACMP_FILTER_LEN32))

#if defined (DPM32M036)
#define IS_ACMP_P_INPUT(x) (((x) == ACMP_IN_P_SEL_ACMP_IN0)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN1)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN2)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN3)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN4)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN5)  || \
                            ((x) == ACMP_IN_P_SEL_PGA0)      || \
                            ((x) == ACMP_IN_P_SEL_PGA1)      || \
                            ((x) == ACMP_IN_P_SEL_PGA2)      || \
                            ((x) == ACMP_IN_P_SEL_OCP))
#elif defined (DPM32M030)
#define IS_ACMP_P_INPUT(x) (((x) == ACMP_IN_P_SEL_ACMP_IN0)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN1)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN2)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN3)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN4)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN5)  || \
                            ((x) == ACMP_IN_P_SEL_PGA0)      || \
                            ((x) == ACMP_IN_P_SEL_PGA1)      || \
                            ((x) == ACMP_IN_P_SEL_OCP))
#elif defined (DPM32M015)
#define IS_ACMP_P_INPUT(x) (((x) == ACMP_IN_P_SEL_ACMP_IN0)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN1)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN2)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN3)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN4)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN5)  || \
                            ((x) == ACMP_IN_P_SEL_PGA0)      || \
                            ((x) == ACMP_IN_P_SEL_OCP))                      
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_ACMP_P_INPUT(x) (((x) == ACMP_IN_P_SEL_ACMP_IN0)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN1)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN2)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN3)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN4)  || \
                            ((x) == ACMP_IN_P_SEL_ACMP_IN5)  || \
                            ((x) == ACMP_IN_P_SEL_PGA0)      || \
                            ((x) == ACMP_IN_P_SEL_PGA1))
#endif /* DPM32M036 */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
#define IS_ACMP_N_INPUT(x) (((x) == ACMP_IN_N_SEL_ACMP_IN0)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN1)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN2)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN3)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN4)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN5)  || \
                            ((x) == ACMP_IN_N_SEL_VREF_DIV))
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_ACMP_N_INPUT(x) (((x) == ACMP_IN_N_SEL_ACMP_IN0)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN1)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN2)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN3)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN4)  || \
                            ((x) == ACMP_IN_N_SEL_ACMP_IN5)  || \
                            ((x) == ACMP_IN_N_SEL_DAC_OUT)   || \
                            ((x) == ACMP_IN_N_SEL_VREF_DIV))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */



/* ACMP configuration functions. */
void                    ACMP_COMDeInit(void);
void                    ACMP_ResSelect(ACMP_ResSelEnum ACMP_Res);
void                    ACMP_VrefSelect(ACMP_VrefSelEnum ACMP_Vref);

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x) || defined (DPM32M036) || defined (DPM32M030)
void                    ACMP_BEMFRCmd(FunctionalState NewState);
#endif /* DPM32M08x || DPM32M05x || DPM32M03x || DPM32M036 || DPM32M030 */

#if defined (DPM32M036) || defined (DPM32M030)
void                    ACMP_BEMFInputSelect(ACMP_BEMFRInput0Enum ACMP_BEMFRInput0,
                                             ACMP_BEMFRInput1Enum ACMP_BEMFRInput1, 
                                             ACMP_BEMFRInput2Enum ACMP_BEMFRInput2);
#endif /* DPM32M036 || DPM32M030 */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)
void                    ACMP_OCPInputSelect(ACMP_OCPInputEnum ACMP_OCPInput);
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/* Deinitializes and initialization and cmd functions. */
void                    ACMP_DeInit(ACMP_ChannelEnum ACMP_Channel);
void                    ACMP_Init(ACMP_ChannelEnum ACMP_Channel, ACMP_InitTypeStruct* ACMP_InitType);
void                    ACMP_StructInit(ACMP_InitTypeStruct* ACMP_InitType);
void                    ACMP_Cmd(ACMP_ChannelEnum ACMP_Channel, FunctionalState NewState);
void                    ACMP_FilterInit(ACMP_ChannelEnum ACMP_Channel, ACMP_FilterInitTypeStruct* ACMP_FilterInitType);
void                    ACMP_FilterStructInit(ACMP_FilterInitTypeStruct* ACMP_FilterInitType);
void                    ACMP_FilterCmd(ACMP_ChannelEnum ACMP_Channel, FunctionalState NewState);

/* Get ACMP status register value functions. */
FlagState               ACMP_GetCompareResult(ACMP_ChannelEnum ACMP_Channel);
FlagState               ACMP_GetReadyStatus(ACMP_ChannelEnum ACMP_Channel);

/*ACMP interrupt functions. */
void                    ACMP_IntCmd(ACMP_ChannelEnum ACMP_Channel, ACMP_IntTypeEnum ACMP_IntType, FunctionalState NewState);
FunctionalState         ACMP_GetIntCmdStatus(ACMP_ChannelEnum ACMP_Channel);
FlagState               ACMP_GetIntFlagStatus(ACMP_ChannelEnum ACMP_Channel);
void                    ACMP_ClearIntFlag(ACMP_ChannelEnum ACMP_Channel);



#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_ACMP_H__ */
