/**
 *******************************************************************************
 * @file    dpm32m0xx_dsp.h
 *
 * @brief   Header file for DSP firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_DSP_H__
#define __DPM32M0XX_DSP_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"

#if defined (DPM32M08x)

/**
 ******************************************************************************
 * @brief DSP interrupt type enumeration.
 *****************************************************************************/
typedef enum
{
  DSP_INT_TYPE_DIV_ZERO          = 0x00002000UL,   /*!< Div zero interrupt.         */
  DSP_INT_TYPE_UDR               = 0x00004000UL,   /*!< Repeated trigger interrupt. */
  DSP_INT_TYPE_INS_ERR           = 0x00008000UL,   /*!< Instruct error interrupt.   */
  DSP_INT_TYPE_DONE              = 0x40000000UL    /*!< Run completed interrupt.    */
} DSP_IntTypeEnum;


/**
 ******************************************************************************
 * @brief DSP interrupt flag enumeration.
 *****************************************************************************/
typedef enum
{
  DSP_INT_FLAG_DIV_ZERO          = 0x00020000UL,   /*!< DSP div zero interrupt flag.          */
  DSP_INT_FLAG_DONE              = 0x00040000UL,   /*!< DSP run completed interrupt flag.     */
  DSP_INT_FLAG_WDG               = 0x00080000UL,   /*!< DSP watchdog overflow interrupt flag. */
  DSP_INT_FLAG_BKPT              = 0x00100000UL,   /*!< DSP break point interrupt flag.       */
  DSP_INT_FLAG_UDR               = 0x00200000UL,   /*!< DSP repeated trigger flag.            */
  DSP_INT_FLAG_INS_ERR           = 0x00400000UL    /*!< DSP instruct error interrupt flag.    */
} DSP_IntFlagEnum;

/**
 ******************************************************************************
 * @brief DSP DMA trig source enumeration.
 *****************************************************************************/
typedef enum
{
  DSP_DMA_CH0                    = 0x00000000UL,   /*!< DMA CH0 transmission completion triggers operation  */
  DSP_DMA_CH1                    = 0x08000000UL,   /*!< DMA CH1 transmission completion triggers operation  */
  DSP_DMA_CH2                    = 0x10000000UL,   /*!< DMA CH2 transmission completion triggers operation. */
  DSP_DMA_CH3                    = 0x18000000UL,   /*!< DMA CH3 transmission completion triggers operation. */
  DSP_DMA_CH4                    = 0x20000000UL    /*!< DMA CH4 transmission completion triggers operation. */
} DSP_DMAChEnum;

/**
 ******************************************************************************
 * @brief DSP status enumeration.
 *****************************************************************************/
typedef enum
{
  DSP_STATUS_IDLE                = 0x00UL,   /*!< DSP idle .      */
  DSP_STATUS_RUNNING                         /*!< DSP is running. */
} DSP_RunStatusEnum;

/**
 ******************************************************************************
 * @brief DSP GPRx enumeration.
 *****************************************************************************/
typedef enum
{
  DSP_GPR0                       = 0x00UL,  /*!< DSP GPR0.  */
  DSP_GPR1,                                 /*!< DSP GPR1.  */
  DSP_GPR2,                                 /*!< DSP GPR2.  */
  DSP_GPR3,                                 /*!< DSP GPR3.  */
  DSP_GPR4,                                 /*!< DSP GPR4.  */
  DSP_GPR5,                                 /*!< DSP GPR5.  */
  DSP_GPR6,                                 /*!< DSP GPR6.  */
  DSP_GPR7,                                 /*!< DSP GPR7.  */
  DSP_GPR8,                                 /*!< DSP GPR8.  */
  DSP_GPR9,                                 /*!< DSP GPR9.  */
  DSP_GPR10,                                /*!< DSP GPR10. */
  DSP_GPR11,                                /*!< DSP GPR11. */
  DSP_GPR12,                                /*!< DSP GPR12. */
  DSP_GPR13,                                /*!< DSP GPR13. */
  DSP_GPR14,                                /*!< DSP GPR14. */
  DSP_GPR15                                 /*!< DSP GPR15. */
} DSP_GPRxEnum;

/**
 ******************************************************************************
 * @brief DSP SDRx enumeration.
 *****************************************************************************/
typedef enum
{
  DSP_SDR0                       = 0x00UL,  /*!< DSP SDR0.  */
  DSP_SDR1,                                 /*!< DSP SDR1.  */
  DSP_SDR2,                                 /*!< DSP SDR2.  */
  DSP_SDR3,                                 /*!< DSP SDR3.  */
  DSP_SDR4,                                 /*!< DSP SDR4.  */
  DSP_SDR5,                                 /*!< DSP SDR5.  */
  DSP_SDR6,                                 /*!< DSP SDR6.  */
  DSP_SDR7,                                 /*!< DSP SDR7.  */
  DSP_SDR8,                                 /*!< DSP SDR8.  */
  DSP_SDR9,                                 /*!< DSP SDR9.  */
  DSP_SDR10,                                /*!< DSP SDR10. */
  DSP_SDR11,                                /*!< DSP SDR11. */
  DSP_SDR12,                                /*!< DSP SDR12. */
  DSP_SDR13,                                /*!< DSP SDR13. */
  DSP_SDR14,                                /*!< DSP SDR14. */
  DSP_SDR15,                                /*!< DSP SDR15. */
  DSP_SDR16,                                /*!< DSP SDR16. */
  DSP_SDR17,                                /*!< DSP SDR17. */
  DSP_SDR18,                                /*!< DSP SDR18. */
  DSP_SDR19,                                /*!< DSP SDR19. */
  DSP_SDR20,                                /*!< DSP SDR20. */
  DSP_SDR21,                                /*!< DSP SDR21. */
  DSP_SDR22,                                /*!< DSP SDR22. */
  DSP_SDR23,                                /*!< DSP SDR23. */
  DSP_SDR24,                                /*!< DSP SDR24. */
  DSP_SDR25,                                /*!< DSP SDR25. */
  DSP_SDR26,                                /*!< DSP SDR26. */
  DSP_SDR27,                                /*!< DSP SDR27. */
  DSP_SDR28,                                /*!< DSP SDR28. */
  DSP_SDR29,                                /*!< DSP SDR29. */
  DSP_SDR30,                                /*!< DSP SDR30. */
  DSP_SDR31                                 /*!< DSP SDR31. */
} DSP_SDRxEnum;



/**
 ******************************************************************************
 * @brief DSP initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t          DSP_PcStart;         /*!< DSP running start address.
                                              This parameter can be a number between 0x0000 and 0x1FF.    */
  uint16_t          DSP_PcEnd;           /*!< DSP running end address and must larger than start address.
                                              This parameter can be a number between 0x0000 and 0x1FF.    */
  DSP_DMAChEnum     DSP_DMACh;           /*!< DSP DMA trigger source select.
                                              This parameter can be a value of @ref DSP_DMAChEnum.        */
  FunctionalState   DSP_DMATrigEn;       /*!< The completion of the DMA channel transfer triggers the
                                              DSP to start executing the program.                         */
  FunctionalState   DSP_DMADoneEn;       /*!< Send DMA request when DSP executor completes.               */

} DSP_InitTypeStruct;


/**
 ******************************************************************************
 * @brief DSP watchdog initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t          DSP_WDGTimOutVal;    /*!< Watchdog maximum count value.
                                              This parameter can be a number between 0x0000 and 0x3FF.    */
  FunctionalState   DSP_WDGSystemRstEn;  /*!< Watchdog count exceeds maximum count value,request reset.   */

} DSP_WDGInitTypeStruct;


/* DSP param check. */

#define IS_DSP_DAM_CH(x)           (((x) == DSP_DMA_CH0) || \
                                    ((x) == DSP_DMA_CH1) || \
                                    ((x) == DSP_DMA_CH2) || \
                                    ((x) == DSP_DMA_CH3) || \
                                    ((x) == DSP_DMA_CH4))

#define IS_DSP_PC_START_ADDR(x)    ((x) <= 0x1FFUL)

#define IS_DSP_PC_END_ADDR(x)      ((x) <= 0x1FFUL)



/* Initialization and Configuration functions. */
void                    DSP_DeInit(void);
void                    DSP_Init(DSP_InitTypeStruct* DSP_InitType);
void                    DSP_StructInit(DSP_InitTypeStruct* DSP_InitType);
void                    DSP_Cmd(FunctionalState NewState);

/* DSP runing and interrupt flag functions. */
void                    DSP_SoftTrigStart(void);
uint16_t                DSP_GetPCValue(void);
DSP_RunStatusEnum       DSP_GetRunStatus(void);

/* DSP dma functions. */
void                    DSP_DMATrigCmd(FunctionalState NewState);
void                    DSP_DMADoneCmd(FunctionalState NewState);

/* DSP interrupt functions. */
void                    DSP_IntCmd(DSP_IntTypeEnum DSP_IntType, FunctionalState NewState);
FunctionalState         DSP_GetIntCmdStatus(DSP_IntTypeEnum DSP_IntType);
FlagState               DSP_GetIntFlagStatus(DSP_IntFlagEnum DSP_IntFlag);
void                    DSP_ClearIntFlag(DSP_IntFlagEnum DSP_IntFlag);

/* DSP watchdog functions. */
void                    DSP_WDGInit(DSP_WDGInitTypeStruct* DSP_WDGInitType);
void                    DSP_WDGStructInit(DSP_WDGInitTypeStruct* DSP_WDGInitType);
void                    DSP_WDGCmd(FunctionalState NewState);
void                    DSP_WDGIntCmd(FunctionalState NewState);

/* DSP debug break point functions. */
void                    DSP_DBGBreakpointConfig(uint16_t DSP_DBGBkpValue);
void                    DSP_DBGBreakpointCmd(FunctionalState NewState);
void                    DSP_DBGBreakpointIntCmd(FunctionalState NewState);

/* DSP GPRx SDRx and INSx operate functions. */
void                    DSP_GPRWrite(DSP_GPRxEnum GPRx, int16_t GPR_Data);
int16_t                 DSP_GPRRead(DSP_GPRxEnum GPRx);
void                    DSP_SDRWrite(DSP_SDRxEnum SDRx, int16_t SDR_Data);
int16_t                 DSP_SDRRead(DSP_SDRxEnum SDRx);
void                    DSP_INSWrite(uint8_t INSx, uint32_t INS_Data);
void                    DSP_INSWriteBuffer(uint8_t startINSx, uint32_t *INS_Data, uint16_t InsLen);
uint32_t                DSP_INSRead(uint8_t INSx);
void                    DSP_INSReadBuffer(uint8_t startINSx, uint32_t *INS_Data, uint16_t InsLen);

#endif /* DPM32M08x */

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif


#endif /* __DPM32M0XX_DSP_H__ */
